/*
Car Scrolling Display (with USB port) Source Code
Author: Mauro Grassi, September 2008 for Silicon Chip Publications.
Using three 5x7 dot matrix modules Kingbright TA12-11EWA
Notes: by Mauro Grassi.
USB may lead to crashes of host computer if the bypass capacitor on the Vusb pin is not big enough in value. 
I have been using a 1uF and it works well. With a 100nF instead you will get constant
crashing of host PC when USB plugged in. 
Note that the datasheet recommends 220nF certainly don't use any smaller than this.
Version 5.00 is the first version for on the PC board. Different to the previous versions in the pinouts. The previous versions were on prototyping breadboard.
As of version 7.0 the polynomial interpolation code has been externalized to the PC host program. The firmware only holds the coefficients of the polynomial.
Compiling Instructions: small memory model single bank small data model, must use custom linker script, as follows (with large stack (ie >256 bytes) enabled with
the -ls option to the compiler in the alternative command line options box:

Version 11.60 is the first version with dependent variables
Compile using 1 pass of procedural abstraction using the microchip C18 C compiler

Version 12.20 is the first version with Battery Protection (need to enable battSleepMode, and connect power differently, connect omnipresent power
to power and connect switched ignition power to battery sense!).
*/

#include <p18cxxx.h>
#include "typedefs.h"                    // Required
#include "usb.h"                         // Required
#include "io_cfg.h"                      // Required
#include <math.h>
#include "carscroll.h"

#if   defined(__18F4550)||defined(__18F4455)|| \
	  defined(__18F2550)||defined(__18F2455)|| \
      defined(__18F4553)||defined(__18F4458)|| \
      defined(__18F2553)||defined(__18F2458)
//

#pragma config PLLDIV   = 5       		// (20 MHz input)
#pragma config CPUDIV   = OSC1_PLL2		//
#pragma config USBDIV   = 2       		// Clock source from 96MHz PLL/2
#pragma config FOSC     = HSPLL_HS		//
#pragma config FCMEN    = OFF
#pragma config IESO     = OFF
#pragma config PWRT     = ON
#pragma config BOR      = ON
#pragma config BORV     = 3
#pragma config VREGEN   = ON
#pragma config WDT      = OFF
#pragma config WDTPS    = 32768
#pragma config MCLRE    = ON
#pragma config LPT1OSC  = OFF			// you could enable this if you want.
#pragma config PBADEN   = OFF
#pragma config CCP2MX   = ON
#pragma config STVREN   = ON			// Stack Overflow Reset Enabled!!
#pragma config LVP      = OFF
#pragma config ICPRT    = OFF       	// Dedicated In-Circuit Debug/Programming
#pragma config XINST    = OFF       	// Extended Instruction Set
#pragma config CP0      = OFF
#pragma config CP1      = OFF
#pragma config CP2      = OFF
#pragma config CP3      = OFF
#pragma config CPB      = OFF
#pragma config CPD      = OFF
#pragma config WRT0     = OFF
#pragma config WRT1     = OFF
#pragma config WRT2     = OFF
#pragma config WRT3     = OFF
#pragma config WRTB     = OFF      		// Boot Block Write Protection
#pragma config WRTC     = OFF
#pragma config WRTD     = OFF
#pragma config EBTR0    = OFF
#pragma config EBTR1    = OFF
#pragma config EBTR2    = OFF
#pragma config EBTR3    = OFF
#pragma config EBTRB    = OFF
//
#endif

double readAN(byte);
int doRWRam(ram far byte*, int, byte);
int doRWRom(MEM_PTR, int, byte);
int doRWEE(int, ram byte*, int, byte);
int updateVariable(int);
int myWriteProgMemUpTo32(MEM_PTR, int, int);
int myReadProgMemUpTo32(MEM_PTR, int, int);
void prepareVariableInfo(int, ram byte *, int, int);
void printcharDots(int,int,int);
void putVariable(int);
void getVariable(int);
void saveAllSettings(void);
void restoreAllSettings(void);
void printstringDots(rom char*, int, int, char);
void blankDisplay(void);
/*
  The following are Global Variables
*/

#pragma udata gpr0
char tempbuffer[TEMPBUFF_SIZE];

unsigned int fperl[FCHANNELS];
unsigned int fperh[FCHANNELS];
unsigned int pperl[FCHANNELS];
unsigned int pperh[FCHANNELS];

#pragma udata gpr1
byte pushVariable;
byte pushIndex;
byte pushCount;
byte pushing;
byte pushStart;
int addressVariableFlash;
int addressStringFlash;
byte maxStringSize;
int stringsSize;
byte usb_sense;
//MAPPOINT freezePoint;
byte verhigh;
byte verlow;
byte battWakeUpMode;
double previousValues[NUM_RAW*NUM_VARS];
unsigned int tmr3w, tmr1w;
OUTPUT output[NUM_OUTPUTS];
byte outputNextState[NUM_OUTPUTS];
char screenCounter;
byte displayPhase;
//
// byte displayDir;			// direction of scrolling
//
// These three variables control the display brightness
//
double dt;
byte minorDelay;
byte majorDelay;
byte minBrightness;					// this is the minimum brightness (0-255)
byte screenOnPeriod;				// this is the actual brightness level
byte maxBrightness;					// this is the maximum brightness (0-255)
int ldrLevel;
int ldrPercent10;					// in tens of percent
int battLevel;						// the battery voltage level
/*
byte battState;                     // for tracking transitions in the batt level used for 
                                    // zeroAcc mode =0 when Power is omnipresent on the input and batt sense senses the switched ignition instead of the normal
                                    // connection of power is switched ignition and batt sense is omnipresent.
                                    // battState=0 is the initial state (unknown configuration)
                                    // battState=1 means there was a transition from battLevel=0 to battLevel>0 and battLevel is now >0
                                    // battState=2 means there was a transition from battLevel>0 to battLevel=0 and battLevel is now=0
                                    */
byte battSleepMode;                 // =0 no sleeping when battLevel is below battMin else non-zero: go to sleep when detected that battLevel is low
                                    // note that the higher the non-zero value, the longer delay from the detection to power down to sleep!
                                    // this delay is dependent on the variable update frequency too!
char battLow;                        // battLow=0 when battery below battMin (with delay), else non-zero
byte battScaling;
byte reqReset;
double battVoltage;
byte battMin;
byte battInv;
byte battMax;
byte limitCounter;
byte limitObj;
byte limitPeriod;
byte systemDecp;					// default system decimal points
// These two variables control the scrolling speed
byte scrollSpeed;
byte fmode[FCHANNELS];
// The following are used to debounce the pushbutton
byte buttonIndex;				// for the button state machine (ie the state)
byte buttonReq;
long buttonCounter;
byte checkCounter;
byte checkPeriod;
int buttonPress[2];
int cursor;
byte roundingMode;

byte invert;
byte beepCounter;
byte beepPeriod;
//byte updatingVariable;
byte inHighInterrupt;
byte beep;
byte reqBeep;				// request beep
byte localReqBeep;
byte szVariable;
byte currentUpdateVariable;
int fosc4div1000;						// in KHz
byte imode;								// current imode
byte rimode;							// requested imode
byte zeroAcc;
byte invertMask;
byte tmr0Con;
byte tmr0ConLog;
byte tmr0HPeriod;						// used to fine tune the variable update frequency!
byte display[DISPLAY_COLUMNS_AND_1*2];	// columns bits	  2 phases for buffer swapping


char buffer[NUM_DIGITS+3];			
byte dummyautosizevar1;				    // this is only for making the auto sizing from the map table work (end dummy variable, not actually used)

#pragma udata gpr2

int systemName;
byte displayMode;
byte displayObject;
long tmr0Ticks;
VARIABLE variable[NUM_VARS];
double polyMemory[MAX_POLY];			// polynomial Memory
byte polyTable[MAX_POLY_INDEXES];
DISPLAYOBJECT displayObj[NUM_DISPLAY_OBJECTS];
byte dummyautosizevar2;				// this is only for making the auto sizing from the map table work (end dummy variable, not actually used)
#pragma code

#pragma code _HIGH_INTERRUPT_VECTOR = 0x000008
void _high_ISR (void)
{
    _asm goto myISRHigh _endasm
	//_asm goto RM_HIGH_INTERRUPT_VECTOR _endasm
}

#pragma code _LOW_INTERRUPT_VECTOR = 0x000018
void _low_ISR (void)
{
    _asm goto myISR _endasm
	//_asm goto RM_LOW_INTERRUPT_VECTOR _endasm
}
//
#pragma code
//
/** D E C L A R A T I O N S **************************************************/
//
void DelayMs(unsigned int delay)
{
	unsigned int n, m;

	if((reqReset!=0)|| ( (imode!=FLASHWRITE_MODE) && ( (pushing==1)|| ( (pushing==0)&&( (buttonReq==1)|| ((rimode & 0x80)!=0) ) ) ) ))
	{
	blankDisplay();
	return;
	}
 	while(delay--)
	{
	for(n = 0; n < 500; n++) 
	{
	// the following must be around a 2us delay...	
	_asm
		nop
		nop
		nop
		nop
	_endasm
	}	
   }
}

void Delay16Ms(unsigned int delay)
{
	// same as delay ms except the argument is in 16ms units
	delay=(delay<<4);
	DelayMs(delay);
}

/*
void wait(void)
{
	_asm
		nop
	_endasm
}
*/

const rom byte SmallFont[]=
{    	
/*
		0x00,0x00,0x00,0x00,		// 20
		0x00,0x00,0x00,0x00,	   	// 21
     	0x80,0x80,0x80,0x80,		// 22
		0x00,0x00,0x00,0x00,	   	// 23 
     	0x00,0x00,0x00,0x00,		// 24 
		0x00,0x00,0x00,0x00,	   	// 25 
     	0x00,0x00,0x00,0x00,		// 26
		0x00,0x00,0x00,0x00,	   	// 27
     	0x00,0x00,0x00,0x00,		// 28
		0x00,0x00,0x00,0x00,   		// 29
     	0x00,0x00,0x00,0x00,		// 2A
		0x00,0x00,0x00,0x00,		// 2B
		0x00,0x00,0x00,0x00,		// 2C
*/
		0x04,0x04,0x04,0x00,	   	// 2D
     	0x80,0x01,0x00,0x80,		// 2E
		0x04,0x04,0x04,0x00,   		// 2F     	
		0x1F,0x11,0x1F,0x00,		// 30
		0x00,0x00,0x1F,0x00, 		// 31
	   	0x17,0x15,0x1D,0x00,		// 32
		0x15,0x15,0x1F,0x00, 		// 33
     	0x1C,0x04,0x1F,0x00, 		// 34
		0x1D,0x15,0x17,0x00,		// 35
     	0x1F,0x15,0x17,0x00,		// 36
		0x10,0x10,0x1F,0x00,   		// 37
     	0x1F,0x15,0x1F,0x00,		// 38
		0x1D,0x15,0x1F,0x00,   		// 39 
     	0x04,0x0E,0x04,0x00			// 3A	 this is used to indicate an overflow!
};

/*

		0x00,0x00,0x00,0x00,		// 3B
		0x00,0x00,0x00,0x00,		// 3C
		0x00,0x00,0x00,0x00,	   	// 3D
     	0x00,0x00,0x00,0x00,		// 3E
		0x00,0x00,0x00,0x00,   		// 3F

		0x00,0x00,0x00,0x00,		// 40
		0x1F,0x14,0x1F,0x00,   		// 41
     	0x1F,0x15,0x0A,0x00,		// 42
		0x1F,0x11,0x11,0x00,   		// 43
     	0x1F,0x11,0x0E,0x00,		// 44
		0x1F,0x15,0x15,0x00,	   	// 45
     	0x1F,0x14,0x14,0x00			// 46
};
/*
		0x00,0x00,0x00,0x00,	   	// 47
     	0x00,0x00,0x00,0x00,		// 48
		0x00,0x00,0x00,0x00,	   	// 49
     	0x00,0x00,0x00,0x00,		// 4A
		0x11,0x0A,0x04,0x1F,		// K
		0x00,0x01,0x01,0x1F,		// L
		0x1F,0x08,0x08,0x1F,	   	// M
     	0x00,0x00,0x00,0x00,		// N
		0x00,0x00,0x00,0x00,	   	// O
     	
		0x00,0x00,0x00,0x00,		// P
		0x00,0x00,0x00,0x00,	   	// Q
     	0x00,0x1D,0x16,0x1F,		// R
		0x00,0x00,0x00,0x00,	   	// 53
		};
*/

const rom byte BigFont[]=
{ 
/*
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 0x00
0x3E, 0x51, 0x45, 0x51, 0x3E, 0x00, // 0x01
0x3E, 0x6B, 0x7B, 0x6B, 0x3E, 0x00, // 0x02
0x1C, 0x3E, 0x1F, 0x3E, 0x1C, 0x00, // 0x03
0x0C, 0x1E, 0x3F, 0x1E, 0x0C, 0x00, // 0x04
0x06, 0x36, 0x7F, 0x36, 0x06, 0x00, // 0x05
0x0C, 0x1D, 0x3F, 0x1D, 0x0C, 0x00, // 0x06
0x00, 0x00, 0x0C, 0x0C, 0x00, 0x00, // 0x07
0x7F, 0x7F, 0x73, 0x73, 0x7F, 0x7F, // 0x08
0x00, 0x1E, 0x12, 0x12, 0x1E, 0x00, // 0x09
0x7F, 0x61, 0x6D, 0x6D, 0x61, 0x7F, // 0x0A
0x38, 0x36, 0x29, 0x09, 0x06, 0x00, // 0x0B
0x30, 0x4A, 0x4F, 0x4A, 0x30, 0x00, // 0x0C
0x10, 0x20, 0x7E, 0x07, 0x03, 0x00, // 0x0D
0x7E, 0x56, 0x28, 0x3F, 0x03, 0x00, // 0x0E
0x2A, 0x1C, 0x36, 0x1C, 0x2A, 0x00, // 0x0F
0x08, 0x1C, 0x3E, 0x7F, 0x00, 0x00, // 0x10
0x00, 0x7F, 0x3E, 0x1C, 0x08, 0x00, // 0x11
0x14, 0x36, 0x7F, 0x36, 0x14, 0x00, // 0x12
0x00, 0x7D, 0x00, 0x7D, 0x00, 0x00, // 0x13
0x7F, 0x40, 0x7F, 0x48, 0x30, 0x00, // 0x14
0x22, 0x4D, 0x55, 0x59, 0x22, 0x00, // 0x15
0x00, 0x03, 0x03, 0x03, 0x03, 0x00, // 0x16
0x14, 0x36, 0x7F, 0x36, 0x14, 0x00, // 0x17
0x10, 0x30, 0x7F, 0x30, 0x10, 0x00, // 0x18
0x04, 0x06, 0x7F, 0x06, 0x04, 0x00, // 0x19
0x08, 0x1C, 0x3E, 0x08, 0x08, 0x00, // 0x1A
0x08, 0x08, 0x3E, 0x1C, 0x08, 0x00, // 0x1B
0x01, 0x01, 0x01, 0x01, 0x0F, 0x00, // 0x1C
0x08, 0x3E, 0x08, 0x3E, 0x08, 0x00, // 0x1D
0x06, 0x1E, 0x7E, 0x1E, 0x06, 0x00, // 0x1E
0x60, 0x78, 0x7E, 0x78, 0x60, 0x00, // 0x1F
*/
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 0x20
0x00, 0x30, 0x7D, 0x30, 0x00, 0x00, // 0x21
0x60, 0x70, 0x00, 0x60, 0x70, 0x00, // 0x22
0x12, 0x3F, 0x12, 0x3F, 0x12, 0x00, // 0x23
0x00, 0x24, 0x2B, 0x6A, 0x12, 0x00, // 0x24
0x63, 0x13, 0x08, 0x64, 0x63, 0x00, // 0x25
0x05, 0x02, 0x35, 0x49, 0x36, 0x00, // 0x26
0x00, 0x00, 0x60, 0x70, 0x00, 0x00, // 0x27
0x00, 0x41, 0x3E, 0x00, 0x00, 0x00, // 0x28
0x00, 0x00, 0x3E, 0x41, 0x00, 0x00, // 0x29
0x08, 0x3E, 0x1C, 0x3E, 0x08, 0x00, // 0x2A
0x08, 0x08, 0x3E, 0x08, 0x08, 0x00, // 0x2B
0x00, 0x00, 0x03, 0x03, 0x00, 0x00, // 0x2C
0x08, 0x08, 0x08, 0x08, 0x08, 0x00, // 0x2D
0x80, 0x00, 0x03, 0x03, 0x00, 0x80, // 0x2E
0x20, 0x10, 0x08, 0x04, 0x02, 0x00, // 0x2F
0x3E, 0x51, 0x49, 0x45, 0x3E, 0x00, // 0x30
0x00, 0x01, 0x7F, 0x21, 0x00, 0x00, // 0x31
0x31, 0x49, 0x49, 0x45, 0x23, 0x00, // 0x32
0x36, 0x49, 0x49, 0x49, 0x22, 0x00, // 0x33
0x04, 0x7F, 0x24, 0x14, 0x0C, 0x00, // 0x34
0x46, 0x49, 0x49, 0x49, 0x7A, 0x00, // 0x35
0x06, 0x49, 0x49, 0x29, 0x1E, 0x00, // 0x36
0x60, 0x50, 0x48, 0x47, 0x40, 0x00, // 0x37
0x36, 0x49, 0x49, 0x49, 0x36, 0x00, // 0x38
0x3C, 0x4A, 0x49, 0x49, 0x30, 0x00, // 0x39
0x00, 0x00, 0x1B, 0x1B, 0x00, 0x80, // 0x3A
0x00, 0x00, 0x1B, 0x1B, 0x00, 0x00, // 0x3B
0x00, 0x41, 0x22, 0x14, 0x08, 0x00, // 0x3C
0x12, 0x12, 0x12, 0x12, 0x12, 0x00, // 0x3D
0x08, 0x14, 0x22, 0x41, 0x00, 0x00, // 0x3E
0x30, 0x48, 0x4D, 0x40, 0x20, 0x00, // 0x3F
0x3C, 0x55, 0x5D, 0x41, 0x3E, 0x00, // 0x40
0x3F, 0x44, 0x44, 0x44, 0x3F, 0x00, // 0x41
0x36, 0x49, 0x49, 0x49, 0x7F, 0x00, // 0x42
0x22, 0x41, 0x41, 0x41, 0x3E, 0x00, // 0x43
0x3E, 0x41, 0x41, 0x41, 0x7F, 0x00, // 0x44
0x41, 0x49, 0x49, 0x49, 0x7F, 0x00, // 0x45
0x40, 0x48, 0x48, 0x48, 0x7F, 0x00, // 0x46
0x2F, 0x49, 0x49, 0x41, 0x3E, 0x00, // 0x47
0x7F, 0x08, 0x08, 0x08, 0x7F, 0x00, // 0x48
0x00, 0x41, 0x7F, 0x41, 0x00, 0x00, // 0x49
0x7E, 0x01, 0x01, 0x01, 0x06, 0x00, // 0x4A
0x41, 0x22, 0x14, 0x08, 0x7F, 0x00, // 0x4B
0x01, 0x01, 0x01, 0x01, 0x7F, 0x00, // 0x4C
0x7F, 0x20, 0x10, 0x20, 0x7F, 0x00, // 0x4D
0x7F, 0x08, 0x10, 0x20, 0x7F, 0x00, // 0x4E
0x3E, 0x41, 0x41, 0x41, 0x3E, 0x00, // 0x4F
0x30, 0x48, 0x48, 0x48, 0x7F, 0x00, // 0x50
0x3D, 0x42, 0x45, 0x41, 0x3E, 0x00, // 0x51
0x33, 0x4C, 0x48, 0x48, 0x7F, 0x00, // 0x52
0x26, 0x49, 0x49, 0x49, 0x32, 0x00, // 0x53
0x40, 0x40, 0x7F, 0x40, 0x40, 0x00, // 0x54
0x7E, 0x01, 0x01, 0x01, 0x7E, 0x00, // 0x55
0x7C, 0x02, 0x01, 0x02, 0x7C, 0x00, // 0x56
0x7E, 0x01, 0x1E, 0x01, 0x7E, 0x00, // 0x57
0x63, 0x14, 0x08, 0x14, 0x63, 0x00, // 0x58
0x70, 0x08, 0x07, 0x08, 0x70, 0x00, // 0x59
0x00, 0x61, 0x51, 0x49, 0x47, 0x00, // 0x5A
0x00, 0x41, 0x41, 0x7F, 0x00, 0x00, // 0x5B
0x02, 0x04, 0x08, 0x10, 0x20, 0x00, // 0x5C
0x00, 0x7F, 0x41, 0x41, 0x00, 0x00, // 0x5D
0x10, 0x20, 0x40, 0x20, 0x10, 0x00, // 0x5E
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 0x5F
0x00, 0x00, 0x70, 0x60, 0x00, 0x00, // 0x60
0x0F, 0x15, 0x15, 0x15, 0x02, 0x00, // 0x61 = a
0x0E, 0x11, 0x11, 0x11, 0x7F, 0x00, // 0x62 = b
0x0A, 0x11, 0x11, 0x11, 0x0E, 0x00, // 0x63 = c
0x7F, 0x11, 0x11, 0x11, 0x0E, 0x00, // 0x64 = d
0x08, 0x15, 0x15, 0x15, 0x0E, 0x00, // 0x65 = e
0x00, 0x48, 0x48, 0x3F, 0x08, 0x00, // 0x66 = f
0x1E, 0x25, 0x25, 0x25, 0x18, 0x00, // 0x67 = g (*)
0x00, 0x0F, 0x10, 0x10, 0x7F, 0x00, // 0x68 = h
0x00, 0x01, 0x5F, 0x00, 0x00, 0x00, // 0x69 = i
0x00, 0x5E, 0x11, 0x11, 0x02, 0x00, // 0x6A = j (*)
0x00, 0x11, 0x0A, 0x04, 0x7F, 0x00, // 0x6B = k
0x00, 0x01, 0x7F, 0x00, 0x00, 0x00, // 0x6C = l
0x0F, 0x10, 0x0C, 0x10, 0x1F, 0x00, // 0x6D = m
0x00, 0x0F, 0x10, 0x10, 0x1F, 0x00, // 0x6E = n
0x0E, 0x11, 0x11, 0x11, 0x0E, 0x00, // 0x6F = o
0x0C, 0x12, 0x12, 0x12, 0x1F, 0x00, // 0x70 = p (*)
0x1F, 0x12, 0x12, 0x12, 0x0C, 0x00, // 0x71 = q (*)
0x08, 0x10, 0x11, 0x0F, 0x11, 0x00, // 0x72 = r
0x02, 0x15, 0x15, 0x15, 0x08, 0x00, // 0x73 = s
0x00, 0x12, 0x11, 0x3E, 0x10, 0x00, // 0x74 = t
0x00, 0x1F, 0x02, 0x01, 0x1E, 0x00, // 0x75 = u
0x1C, 0x02, 0x01, 0x02, 0x1C, 0x00, // 0x76 = v
0x1E, 0x03, 0x06, 0x03, 0x1E, 0x00, // 0x77 = w
0x00, 0x1B, 0x04, 0x04, 0x1B, 0x00, // 0x78 = x
0x00, 0x3E, 0x09, 0x09, 0x32, 0x00, // 0x79 = y (*)
0x00, 0x19, 0x15, 0x15, 0x13, 0x00 // 0x7A = z 
};

/*
0x00, 0x41, 0x41, 0x3E, 0x08, 0x00, // 0x7B
0x00, 0x00, 0x77, 0x00, 0x00, 0x00, // 0x7C
0x08, 0x3E, 0x41, 0x41, 0x00, 0x00, // 0x7D
0x00, 0x40, 0x20, 0x40, 0x20, 0x00, // 0x7E
0x1E, 0x32, 0x62, 0x32, 0x1E, 0x00 // 0x7F
};

0x00, 0x00, 0x70, 0x60, 0x00, 0x00, // 0x80
0x00, 0x5F, 0x02, 0x01, 0x5E, 0x00, // 0x81
0x00, 0x00, 0x70, 0x60, 0x00, 0x00, // 0x82
0x00, 0x00, 0x70, 0x60, 0x00, 0x00, // 0x83
0x00, 0x00, 0x70, 0x60, 0x00, 0x00, // 0x84
0x00, 0x00, 0x70, 0x60, 0x00, 0x00, // 0x85
0x00, 0x00, 0x70, 0x60, 0x00, 0x00, // 0x86
0x00, 0x00, 0x70, 0x60, 0x00, 0x00, // 0x87
0x00, 0x00, 0x70, 0x60, 0x00, 0x00, // 0x88
0x00, 0x00, 0x70, 0x60, 0x00, 0x00, // 0x89
0x00, 0x00, 0x70, 0x60, 0x00, 0x00, // 0x8A
0x00, 0x00, 0x70, 0x60, 0x00, 0x00, // 0x8B
0x00, 0x00, 0x70, 0x60, 0x00, 0x00, // 0x8C
0x00, 0x01, 0x1F, 0x40, 0x00, 0x00, // 0x8D
0x00, 0x00, 0x70, 0x60, 0x00, 0x00, // 0x8E
0x0F, 0x7A, 0x52, 0x7A, 0x0F, 0x00, // 0x8F
0x51, 0x55, 0x15, 0x15, 0x1F, 0x00, // 0x90
0x00, 0x00, 0x70, 0x60, 0x00, 0x00, // 0x91
0x00, 0x00, 0x70, 0x60, 0x00, 0x00, // 0x92
0x00, 0x00, 0x70, 0x60, 0x00, 0x00, // 0x93
0x00, 0x00, 0x70, 0x60, 0x00, 0x00, // 0x94
0x00, 0x00, 0x70, 0x60, 0x00, 0x00, // 0x95
0x00, 0x00, 0x70, 0x60, 0x00, 0x00, // 0x96
0x00, 0x00, 0x70, 0x60, 0x00, 0x00, // 0x97
0x00, 0x00, 0x70, 0x60, 0x00, 0x00, // 0x98
0x00, 0x00, 0x70, 0x60, 0x00, 0x00, // 0x99
0x00, 0x00, 0x70, 0x60, 0x00, 0x00, // 0x9A
0x00, 0x00, 0x70, 0x60, 0x00, 0x00, // 0x9B
0x00, 0x00, 0x70, 0x60, 0x00, 0x00, // 0x9C
0x4A, 0x2A, 0x1F, 0x2A, 0x4A, 0x00, // 0x9D
0x00, 0x00, 0x70, 0x60, 0x00, 0x00, // 0x9E
0x00, 0x00, 0x70, 0x60, 0x00, 0x00, // 0x9F
0x0F, 0x55, 0x55, 0x15, 0x02, 0x00, // 0xA0
0x00, 0x41, 0x5F, 0x00, 0x00, 0x00, // 0xA1
0x00, 0x4E, 0x51, 0x11, 0x0E, 0x00, // 0xA2
0x00, 0x5F, 0x42, 0x01, 0x1E, 0x00, // 0xA3
0x00, 0x47, 0x28, 0x48, 0x2F, 0x00, // 0xA4
0x00, 0x4F, 0x22, 0x44, 0x2F, 0x00, // 0xA5
0x3D, 0x55, 0x55, 0x55, 0x08, 0x00, // 0xA6
0x00, 0x39, 0x45, 0x45, 0x39, 0x00, // 0xA7
0x02, 0x01, 0x59, 0x09, 0x06, 0x00, // 0xA8
0x10, 0x10, 0x10, 0x10, 0x1E, 0x00, // 0xA9
0x1C, 0x10, 0x10, 0x10, 0x10, 0x10, // 0xAA
0x05, 0x2B, 0x19, 0x08, 0x74, 0x00, // 0xAB
0x0F, 0x2A, 0x16, 0x08, 0x74, 0x00, // 0xAC
0x00, 0x06, 0x5F, 0x06, 0x00, 0x00, // 0xAD
0x14, 0x08, 0x00, 0x14, 0x08, 0x00, // 0xAE
0x08, 0x1C, 0x3E, 0x7F, 0x00, 0x00, // 0xAF
0x44, 0x11, 0x44, 0x11, 0x44, 0x11, // 0xB0
0x55, 0x2A, 0x55, 0x2A, 0x55, 0x2A, // 0xB1
0x3B, 0x6E, 0x3B, 0x6E, 0x3B, 0x6E, // 0xB2
0x00, 0x00, 0x7F, 0x00, 0x00, 0x00, // 0xB3
0x00, 0x00, 0x7F, 0x08, 0x08, 0x08, // 0xB4
0x00, 0x00, 0x7F, 0x28, 0x28, 0x28, // 0xB5
0x00, 0x00, 0x7F, 0x00, 0x7F, 0x08, // 0xB6
0x00, 0x00, 0x0F, 0x08, 0x0F, 0x08, // 0xB7
0x00, 0x00, 0x3F, 0x28, 0x28, 0x28, // 0xB8
0x00, 0x00, 0x7F, 0x00, 0x6F, 0x28, // 0xB9
0x00, 0x00, 0x7F, 0x00, 0x7F, 0x00, // 0xBA
0x00, 0x00, 0x3F, 0x20, 0x2F, 0x28, // 0xBB
0x00, 0x00, 0x78, 0x08, 0x68, 0x28, // 0xBC
0x00, 0x00, 0x78, 0x08, 0x78, 0x08, // 0xBD
0x00, 0x00, 0x78, 0x28, 0x28, 0x28, // 0xBE
0x00, 0x00, 0x0F, 0x08, 0x08, 0x08, // 0xBF
0x08, 0x08, 0x78, 0x00, 0x00, 0x00, // 0xC0
0x08, 0x08, 0x78, 0x08, 0x08, 0x08, // 0xC1
0x08, 0x08, 0x0F, 0x08, 0x08, 0x08, // 0xC2
0x08, 0x08, 0x7F, 0x00, 0x00, 0x00, // 0xC3
0x08, 0x08, 0x08, 0x08, 0x08, 0x08, // 0xC4
0x08, 0x08, 0x7F, 0x08, 0x08, 0x08, // 0xC5
0x28, 0x28, 0x7F, 0x00, 0x00, 0x00, // 0xC6
0x08, 0x08, 0x7F, 0x00, 0x7F, 0x00, // 0xC7
0x28, 0x28, 0x68, 0x08, 0x78, 0x00, // 0xC8
0x28, 0x28, 0x2F, 0x20, 0x3F, 0x00, // 0xC9
0x28, 0x28, 0x68, 0x08, 0x68, 0x28, // 0xCA
0x28, 0x28, 0x2F, 0x20, 0x2F, 0x28, // 0xCB
0x28, 0x28, 0x6F, 0x00, 0x7F, 0x00, // 0xCC
0x28, 0x28, 0x28, 0x28, 0x28, 0x28, // 0xCD
0x28, 0x28, 0x6F, 0x00, 0x6F, 0x28, // 0xCE
0x28, 0x28, 0x68, 0x28, 0x28, 0x28, // 0xCF
0x08, 0x08, 0x78, 0x08, 0x78, 0x08, // 0xD0
0x28, 0x28, 0x2F, 0x28, 0x28, 0x28, // 0xD1
0x08, 0x08, 0x0F, 0x08, 0x0F, 0x08, // 0xD2
0x08, 0x08, 0x78, 0x08, 0x78, 0x00, // 0xD3
0x28, 0x28, 0x78, 0x00, 0x00, 0x00, // 0xD4
0x28, 0x28, 0x3F, 0x00, 0x00, 0x00, // 0xD5
0x08, 0x08, 0x0F, 0x08, 0x0F, 0x00, // 0xD6
0x08, 0x08, 0x7F, 0x00, 0x7F, 0x08, // 0xD7
0x28, 0x28, 0x6F, 0x28, 0x28, 0x28, // 0xD8
0x00, 0x00, 0x78, 0x08, 0x08, 0x08, // 0xD9
0x08, 0x08, 0x0F, 0x00, 0x00, 0x00, // 0xDA
0x7F, 0x7F, 0x7F, 0x7F, 0x7F, 0x7F, // 0xDB
0x07, 0x07, 0x07, 0x07, 0x07, 0x07, // 0xDC
0x00, 0x00, 0x00, 0x7F, 0x7F, 0x7F, // 0xDD
0x7F, 0x7F, 0x7F, 0x00, 0x00, 0x00, // 0xDE
0x78, 0x78, 0x78, 0x78, 0x78, 0x78, // 0xDF
0x12, 0x0C, 0x12, 0x12, 0x0C, 0x00, // 0xE0
0x00, 0x16, 0x29, 0x29, 0x3F, 0x00, // 0xE1
0x00, 0x60, 0x40, 0x40, 0x7F, 0x00, // 0xE2
0x20, 0x3F, 0x20, 0x3F, 0x20, 0x00, // 0xE3
0x00, 0x63, 0x49, 0x55, 0x63, 0x00, // 0xE4
0x10, 0x1C, 0x12, 0x12, 0x0C, 0x00, // 0xE5
0x00, 0x1C, 0x02, 0x02, 0x1F, 0x00, // 0xE6
0x00, 0x10, 0x0F, 0x10, 0x08, 0x00, // 0xE7
0x08, 0x55, 0x77, 0x55, 0x08, 0x00, // 0xE8
0x00, 0x3E, 0x49, 0x49, 0x3E, 0x00, // 0xE9
0x19, 0x27, 0x20, 0x27, 0x19, 0x00, // 0xEA
0x00, 0x06, 0x4D, 0x55, 0x22, 0x00, // 0xEB
0x0C, 0x12, 0x0C, 0x12, 0x0C, 0x00, // 0xEC
0x0C, 0x12, 0x3F, 0x12, 0x0C, 0x00, // 0xED
0x00, 0x2A, 0x2A, 0x2A, 0x1C, 0x00, // 0xEE
0x00, 0x1E, 0x20, 0x20, 0x1E, 0x00, // 0xEF
0x00, 0x2A, 0x2A, 0x2A, 0x2A, 0x00, // 0xF0
0x00, 0x12, 0x3A, 0x12, 0x00, 0x00, // 0xF1
0x00, 0x11, 0x29, 0x29, 0x45, 0x00, // 0xF2
0x00, 0x45, 0x29, 0x29, 0x11, 0x00, // 0xF3
0x10, 0x20, 0x1F, 0x00, 0x00, 0x00, // 0xF4
0x00, 0x00, 0x7E, 0x01, 0x02, 0x00, // 0xF5
0x08, 0x08, 0x2A, 0x08, 0x08, 0x00, // 0xF6
0x00, 0x24, 0x12, 0x24, 0x12, 0x00, // 0xF7
0x00, 0x30, 0x48, 0x48, 0x30, 0x00, // 0xF8
0x00, 0x00, 0x0C, 0x0C, 0x00, 0x00, // 0xF9
0x00, 0x00, 0x00, 0x08, 0x00, 0x00, // 0xFA
0x20, 0x20, 0x3E, 0x01, 0x06, 0x00, // 0xFB
0x00, 0x38, 0x40, 0x38, 0x40, 0x00, // 0xFC
0x00, 0x00, 0x28, 0x58, 0x48, 0x00, // 0xFD
0x00, 0x1E, 0x1E, 0x1E, 0x1E, 0x00, // 0xFE
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, // 0xFF
};
*/

rom char emptyString[]={ 'V','a','r', STRING_DELIMITER,'U','n','i','t','\0' };
//
// enforce 32 byte boundary
//
#pragma romdata variableflash=0x300

rom char VariableData[32*SIZE_OF_VARIABLE_TO_32_BYTE_BOUNDARY*NUM_VARS];

#pragma romdata stringflash=0x800

rom char Strings[SIZE_OF_STRINGS];

#pragma code

#if (USE_CLIPPING==1)
void clip(int* p, int min, int max)
{
	if((*p)<min)*p=min; else if((*p)>max)*p=max;
}
#endif

/*
int getIndex(int varn, int index)
{
	// index is 0 to NUM_STRINGS_PER_VAR
	// varn = variable number
#if (USE_CLIPPING==1)
	clip(&index, 0, NUM_STRINGS_PER_VAR-1);
	clip(&varn, 0, NUM_VARS-1);
#endif
	return ((varn * NUM_STRINGS_PER_VAR + index) * maxStringSize);
}

void putString(int varn, int index)
{
	// the string is in dataPacket.data[0-len]
	// the length of the string is in dataPacket.len
	int i;
	MEM_PTR ptr;

#if (USE_CLIPPING==1)
	clip(&index, 0, NUM_STRINGS_PER_VAR-1);
	clip(&varn, 0, NUM_VARS-1);
#endif
	variable[varn].name[index]=(rom char*)&emptyString[0];		// make invisible while updating!
	i=getIndex(varn, index);
	ptr.pAdr=(rom char*)&Strings[i];
	dataPacket.data[dataPacket.len]='\0';
	myWriteProgMemUpTo32(ptr, 0, dataPacket.len);
	variable[varn].name[index]=(rom char*)&Strings[i];
	putVariable(varn);
}
*/

/*
void putVariableMode(int varn, int mmode)
{
#if (USE_CLIPPING==1)
	clip(&varn, 0, NUM_VARS-1);
#endif
	variable[varn].mode=mmode;
	putVariable(varn);
}
*/
/*
void putVariableUpdatePeriod(int varn, int uperiod)
{
	if(varn<0)varn=0; else if(varn>=NUM_VARS)varn=NUM_VARS-1;
	variable[varn].updateper=uperiod;
	variable[varn].iper=uperiod;
	putVariable(varn);
}
*/


void putPolyData(void)
{
	int i;
	byte* ptr;
	for(i=0; i<MAX_POLY_INDEXES;i++)
	{
	WriteEEPROM(POLY_TABLE_ADR+i, polyTable[i]);
	}
	ptr=(byte*)&polyMemory[0];
	for(i=0; i<MAX_POLY*sizeof(double); i++)
	{
	WriteEEPROM(POLY_MEMORY_ADR+i, *ptr++);
	}
}

void getPolyData(void)
{
	int i;
	byte* ptr;
	
	for(i=0; i<MAX_POLY_INDEXES;i++)
	{
	polyTable[i]=ReadEEPROM(POLY_TABLE_ADR+i);
	}
	ptr=(byte *)&polyMemory[0];
	for(i=0; i<MAX_POLY*sizeof(double); i++)
	{
	*ptr++=ReadEEPROM(POLY_MEMORY_ADR+i);
	}
}

/*
void getString(int varn, int index)
{
	// the string will then be in dataPacket.data[0-len]
	// the length will be in dataPacket.len
	int i, x;
	far rom char *ptr;
#if (USE_CLIPPING==1)
	clip(&index, 0, NUM_STRINGS_PER_VAR-1);
	clip(&varn, 0, NUM_VARS-1);
#endif
	ptr=(far rom char*)variable[varn].name[index];
	i=0;
	while((*ptr)!='\0')
	{
	dataPacket.data[i++]=*ptr++;
	}
	dataPacket.data[i]='\0';
	i++;
	dataPacket.len=i;
}
*/

rom char* getStringSubIndex(rom char* ptr, int subindex)
{
	int i;
	i=0;
	while(i<subindex)
	{
		while(((*ptr)!='\0')&&((*ptr)!=STRING_DELIMITER))ptr++;
		if((*ptr)!='\0')ptr++; else break;
		i++;
	}
	return ptr;
}

rom char systemString[]="Battery:V:Ambient Light:%:";

void printString(int varn, int index, int subindex, int font)
{
	rom char* ptr;
	
	if(varn>=NUM_VARS)
	{
	 varn-=NUM_VARS;
	 if((systemName>=0)&&(systemName<SIZE_OF_STRINGS))ptr=&Strings[systemName]; else ptr=&systemString[0];
	 ptr=getStringSubIndex(ptr, subindex+(varn<<1));
	 printstringDots(ptr,font,0, STRING_DELIMITER);
	} 
	else
	{
	 if(variable[varn].name[index]==0)ptr=getStringSubIndex(&emptyString[0], subindex); else ptr=getStringSubIndex(variable[varn].name[index], subindex);
	 printstringDots(ptr,font,0, STRING_DELIMITER);
	 if(variable[varn].name[index]==0)
     { 
	 printcharDots(0x30+varn, font,0);
	 printcharDots('(', font, 0);
	 printcharDots(0x30+index, font,0);
	 printcharDots(')', font, 0);
	 }
  }
}

void putVariable(int varn)
{
	// save variable information to program flash
	MEM_PTR ptr;

#if (USE_CLIPPING==1)
	clip(&varn, 0, NUM_VARS-1);
#endif
	doRWRam((ram far byte*)&variable[varn], sizeof(VARIABLE), READ_DIR);
	ptr.pAdr=(rom far byte*)&VariableData[32*(SIZE_OF_VARIABLE_TO_32_BYTE_BOUNDARY*varn)];
	doRWRom(ptr, sizeof(VARIABLE), WRITE_DIR);
}

void getVariable(int varn)
{
	// get variable information from program flash
	MEM_PTR ptr;

#if (USE_CLIPPING==1)
	clip(&varn, 0, NUM_VARS-1);
#endif
	ptr.pAdr=(rom far byte*)&VariableData[32*(SIZE_OF_VARIABLE_TO_32_BYTE_BOUNDARY*varn)];
	doRWRom(ptr, sizeof(VARIABLE), READ_DIR);
	doRWRam((ram far byte*)&variable[varn], sizeof(VARIABLE), WRITE_DIR);
}

void blankDisplay(void)
{
	int i;
	for(i=0; i<DISPLAY_COLUMNS; i++)display[i+DISPLAY_COLUMNS_AND_1-displayPhase]=0;
	displayPhase=DISPLAY_COLUMNS_AND_1-displayPhase;											// swap buffers!
	cursor=DISPLAY_COLUMNS-1;
}

/*
void loadShiftRegister(byte t)
{
	int i;
	GBIT=1;					// G enable off...
	CKBIT=1;				// CK (rows)=1
	for(i=0; i<8; i++)
	{
	if ((t & 0x80)==0)SERBIT=0; else SERBIT=1;
	CKBIT=0;				// clock
	CKBIT=1;				// clock
	t=t<<1;
	}
	CKBIT=0;				// clock
	CKBIT=1;				// clock
	GBIT=0;
}
*/

/*
void setColumn(int colnum)
{
	switch(colnum)
		{
		case 0:
			PORTAbits.RA4=bitvalue;
			break;
		case 1:
			PORTBbits.RB5=bitvalue;
			break;
		case 2:
			PORTBbits.RB4=bitvalue;
			break;
		case 3:
			PORTBbits.RB3=bitvalue;
			break;
		case 4:
			PORTDbits.RD2=bitvalue;
			break;
		case 5:
			PORTBbits.RB2=bitvalue;
			break;
		case 6:
			PORTBbits.RB0=bitvalue;
			break;
		case 7:
			PORTBbits.RB6=bitvalue;
			break;
		case 8:
			PORTDbits.RD6=bitvalue;
			break;
		case 9:
			PORTBbits.RB7=bitvalue;
			break;
		case 10:
			PORTDbits.RD5=bitvalue;
			break;
		case 11:
			PORTCbits.RC0=bitvalue;
			break;
		case 12:
			PORTDbits.RD0=bitvalue;
			break;
		case 13:
			PORTDbits.RD7=bitvalue;
			break;
		case 14:
			PORTDbits.RD1=bitvalue;
			break;
		default:
			break;
		}
}
*/

void buttonACK(void)
{
	// acknowledge the button press (ie, reset the button sensing system) and reset the beep system too!
	buttonCounter=0;
	buttonIndex=0;
	buttonReq=0;
	beep=0;
	localReqBeep=0;
	reqBeep=0;
	if(((output[0]>>6)& 3)==1)OUTPUT1=0;
	if(((output[1]>>6)& 3)==1)OUTPUT2=0;
}

/*
int isKeyPressed(void)
{
	// returns -1 if short button press
	// returns 1 if long button press
	// return  0 if no key pressed
	int i;
	if(buttonReq==1)
	{
	// note buttonPress are in units of around 40ms so 1 second is equal to 25
	i=(buttonPress[1]-buttonPress[0]);
	if(i<SHORT_BUTTON_PRESS_THRESHOLD)i=SHORT_PRESS; else i=LONG_PRESS;
	return i;
	}
	return NO_PRESS;
}
*/

/*
void isetVar(int value, int* ptr, int minvalue, int maxvalue, byte ee_address)
{
	// int set variable internal
	// sets the variable pointed to by the pointer ptr
	// within the clipping boundary set by minvalue and maxvalue
	// and writes the result to the EEPROM address at ee_address if the value is different to the stored one...
	if(value<minvalue)value=minvalue; else if(value>=maxvalue)value=maxvalue-1;
	*ptr=value;
	WriteEEPROM(ee_address, *ptr);
}
*/

/*
void bsetVar(byte value, byte* ptr, byte minvalue, byte maxvalue, byte ee_address)
{
	// byte set variable internal
	// sets the variable pointed to by the pointer ptr
	// within the clipping boundary set by minvalue and maxvalue
	// and writes the result to the EEPROM address at ee_address if the value is different to the stored one...
	if(value<minvalue)value=minvalue; else if(value>maxvalue)value=maxvalue;
	*ptr=value;
	WriteEEPROM(ee_address, *ptr);
}
*/

void adjustBrightnessLDR(void)
{
	int x;
	// the LDR divider will produce a voltage from 0V to 5.00 * LDR_DIVIDER_RESISTANCE/(LDR_DIVIDER_RESISTANCE+LDR_MIN_RESISTANCE))
	x=(int)(ldrPercent10*0.255);
	if(x<minBrightness)x=minBrightness; 
	if(x>maxBrightness)x=maxBrightness;
	screenOnPeriod=x;
}

void refreshScreen(void)
{
	int i, j;
		if(screenCounter>(DISPLAY_COLUMNS+DISPLAY_COLUMNS-2))screenCounter=0; else screenCounter++;
		if((screenCounter & 0x01)==0)
		{
		i=(screenCounter>>1);
		GBIT=1;
		switch(i)
		{
		case 0:
			PORTDbits.RD1=1;
			PORTAbits.RA4=0;
			break;
		case 1:
			PORTAbits.RA4=1;
			PORTBbits.RB5=0;
			break;
		case 2:
			PORTBbits.RB5=1;
			PORTBbits.RB4=0;
			break;
		case 3:
			PORTBbits.RB4=1;
			PORTBbits.RB3=0;
			break;
		case 4:
			PORTBbits.RB3=1;
			PORTDbits.RD2=0;
			break;
		case 5:
			PORTDbits.RD2=1;
			PORTBbits.RB2=0;
			break;
		case 6:
			PORTBbits.RB2=1;
			PORTBbits.RB0=0;
			break;
		case 7:
			PORTBbits.RB0=1;
			PORTBbits.RB6=0;
			break;
		case 8:
			PORTBbits.RB6=1;
			PORTDbits.RD6=0;
			break;
		case 9:
			PORTDbits.RD6=1;
			PORTBbits.RB7=0;
			break;
		case 10:
			PORTBbits.RB7=1;
			PORTDbits.RD5=0;
			break;
		case 11:
			PORTDbits.RD5=1;
			PORTCbits.RC0=0;
			break;
		case 12:
			PORTCbits.RC0=1;
			PORTDbits.RD0=0;
			break;
		case 13:
			PORTDbits.RD0=1;
			PORTDbits.RD7=0;
			break;
		case 14:
			PORTDbits.RD7=1;
			PORTDbits.RD1=0;
			break;
		default:
			break;
		}
			if(invert==0)j=(display[i+displayPhase] & 0x7F); 
			else 
			if(invert==1)j=(~display[i+displayPhase] & 0x7F); 
			else j=0; 
		//loadShiftRegister(j);
		//GBIT=1;					// G enable off...
		CKBIT=1;				// CK (rows)=1
		for(i=0; i<8; i++)
		{
		if ((j & 0x80)==0)SERBIT=0; else SERBIT=1;
		CKBIT=0;				// clock
		CKBIT=1;				// clock
		j=j<<1;
		}
		CKBIT=0;				// clock
		CKBIT=1;				// clock
		GBIT=0;
		//
		PR2=screenOnPeriod;
	} else { GBIT=1; PR2=255-screenOnPeriod; }
}

void clearUpTo(int start, int limit)
{
	while((start<limit)&&(start<DISPLAY_COLUMNS+1))
	{
	display[start+displayPhase]=0;
	start++;
	}
}

void sendcolumn(byte c)
{
	unsigned int i;
	if((pushing<2)&&(buttonReq==1))
	{
	clearUpTo(0, DISPLAY_COLUMNS+1);
	cursor=DISPLAY_COLUMNS-1;
	return;
	}
	while(cursor>=(DISPLAY_COLUMNS))
		{
		for(i=0;i<(DISPLAY_COLUMNS);i++)
		{
		display[i+DISPLAY_COLUMNS_AND_1-displayPhase]=display[i+1+displayPhase];					// shift display...
		}
		displayPhase=DISPLAY_COLUMNS_AND_1-displayPhase;											// swap buffers!
		DelayMs(scrollSpeed);
		cursor--;
		}
	if(cursor>=-1)
	{ 
	cursor++; display[cursor+displayPhase]=c; // here imode
	if(cursor>=DISPLAY_COLUMNS)DelayMs(scrollSpeed);
	} 
	else cursor++;
}

void printcharDots(int ptr, int font, int maxcols)
{
	int i, j, r;
	unsigned char c;
	int cc;

	cc=ptr;
	if(font==0)
	{
	if(ptr<BIG_FONT_MIN)ptr=BIG_FONT_MIN; else if(ptr>BIG_FONT_MAX)ptr=BIG_FONT_MAX;
	ptr-=BIG_FONT_MIN;
	i=6;
	} else
	{
	if(ptr<SMALL_FONT_MIN)ptr=SMALL_FONT_MIN; else if(ptr>SMALL_FONT_MAX)ptr=SMALL_FONT_MAX;
	ptr-=SMALL_FONT_MIN;
	i=4;
	}
	ptr=ptr*i;
	i--;
	r=0;
	j=i;
	while(j>=0)
	{
	if(font==0)c=BigFont[j+ptr]; else c=(SmallFont[3-j+ptr]);
	if((c& CHAR_BLANKING)==0)r++;
	j--;
	}
	if((maxcols==0)||((r-1+cursor)<=maxcols))
	{
	 while(i>=0)
	 {
	 if(font==0){ c=(BigFont[i+ptr]); } else { c=(SmallFont[3-i+ptr]); }
	 if((((c & CHAR_BLANKING)==0)&&(i>0))||((i==0)&&((maxcols==0)||(cursor<DISPLAY_COLUMNS)))){ if(font==0)sendcolumn(c); else sendcolumn(c<<1); } 
	 i--;
	 }
	}
}

/*
char toUpper(char x)
{
	if((x>='a')&&(x<='z'))return (x-0x20); else return x;
}
*/

void printstringDots(rom char *p, int font, int maxcols, char delimiter)
{
	// with horizontal scrolling... up to maxcols columns output if maxcols=0 just print out till the end...
	char c;
	while(((*p)!='\0')&&((*p)!=delimiter))
	{
	c=*p++;
	if(font==0)printcharDots(c, font, maxcols); else printcharDots(c, font, maxcols);
	}
}

void printstringDotsRam(ram char *p, int font, int maxcols)
{
	// with horizontal scrolling... up to maxcols columns output if maxcols=0 just print out till the end...
	char c;
	
	while((*p)!='\0')
	{
	c=*p++;
	if(font==0)printcharDots(c, font, maxcols); else printcharDots(c, font, maxcols);
	}
}

int getOutputState(double value, double min, double max, byte hysteresis)
{
	// returns the output state corresponding to these values!
	double hyst;
	
	hyst=(double)hysteresis/HYST_SCALE;
	hyst*=(double)HYST_FULLSCALE;
	
	if(value>max)return OS_MAX;
	else
	if(value<min)return OS_MIN;
	else
	if(value>=(max*(1.0-hyst)))return OS_MIDMAX;
	else
	if(value<=(min*(1.0+hyst)))return OS_MIDMIN;
	else
	return OS_MID;
}	

void checkOutputs(int ivarn)
{
	// refresh the outputs
	int j, m;
	byte inv, invd;
	double f;
	
	// first process the visible cues!
	inv=0;	
	if(ivarn<NUM_VARS)
	{
	inv=0;
	for(j=3; j>=0;j--)
	{
	inv=inv<<2;
    f=variable[ivarn].value[j];
	if(f>variable[ivarn].max[j])inv|=0x01;
	else
	if(f<variable[ivarn].min[j])inv|=0x02;
	}
	variable[ivarn].invert=inv & invertMask;	
	} else
	if(ivarn==VAR_BATT)
	{
	f=battVoltage;
	if(f>(battMax*BATTSCALE))inv=1;
	else
	if(f<(battMin*BATTSCALE))inv=2;
	if((inv==2)&&(USBSENSE!=USBSENSEPOLARITY))
	{
    	// approach timeout if no USB power and battery voltage too low
    	if(battLow>0)battLow--; else battLow=0;
    } else 
    {
        if(battSleepMode>0)battLow=battSleepMode; else battLow=DEFAULTBATTSLEEPMODE;
    }    
	battInv=inv & invertMask;
	}
	
	// now update beeps
	m=((output[0]>>3) & 3);
	invd=(inv>>m) & 3;
	j=output[0] & 0x0F;
	m=(output[0]>>6) & 3;
	if((j==ivarn)&&(m==1))
	{
	if(inv!=0)beep |= invd; else beep|=0x04;
	} 	

	m=((output[1]>>3) & 3);
	invd=(inv>>m) & 3;
	j=output[1] & 0x0F;
	m=(output[1]>>6) & 3;
	if((j==ivarn)&&(m==1))
	{
	if(inv!=0)beep |= (invd<<4); else beep|=0x40;
	} 
}

void resetOutputs(void)
{
    int i;
    
    for(i=0; i<NUM_OUTPUTS; i++)
    {
     outputNextState[i]=0;
     if((output[i] & 0xC0)!=0xC0)
     {
        // any mode but relay inverted!
        if(i==0)OUTPUT1=0; else OUTPUT2=0;
     } else
     {        
        // inverted mode
        if(i==0)OUTPUT1=1; else OUTPUT2=1;
     }
    }
}

void checkRelayOutput(int i)
{
	// so now inv=1 if higher than max (1 long beep)
	// and inv=2 if lower than min   
	// and inv=0 otherwise (within range)
		
	// now process the two digital outputs!
	// the output semantic is as follows:
	// bit   	7,6								5,4				  3										        		2,1,0
	//          00 = disabled					= value index	  =0 maximum mode (switching limit is max)				= variable index
	//          01 = buzzer mode								  =1 minimum mode (switching limit is min)
	//          10 = relay normal		(switching switches on!)
	//          11 = relay inverted	    (switching switches off!)
	
	int mode, valueindex, varn, nb, q, oq;
	
	i&=1;
	mode=(output[i]>>6) & 3;
	valueindex=(output[i]>>4) & 3;
	varn=(output[i] & 7);
	if(((varn<NUM_VARS)||(varn==VAR_BATT))&&((mode & 2)!=0))
			{
				if(i==0)oq=OUTPUT1; else oq=OUTPUT2;    		// these are the current states!
				// now oq is the actual state of the output!
				if((output[i] & 0x40)==0)nb=0; else nb=1;		// for inverting the output!
				if(varn<NUM_VARS)
				{
				outputNextState[i]=getOutputState(variable[varn].value[valueindex], variable[varn].min[valueindex], variable[varn].max[valueindex], variable[varn].hysteresis);
				} else
				if(varn==VAR_BATT)
				{
				outputNextState[i]=getOutputState(battVoltage, battMin*BATTSCALE, battMax*BATTSCALE, DEFAULT_BATT_VOLTAGE_HYSTERESIS);
				} 
				q=oq;
				if((output[i] & 0x08)==0)
				{
				// max mode
				if(outputNextState[i]==OS_MAX)
				{ 
				// force output high and force state change!
				q=1 ^ nb; 
				oq=nb;
				}
				else
				if((q!=nb)&&(outputNextState[i]<OS_MIDMAX))
				{
				// force output low and force state change!
				q=nb;
				oq=nb ^ 1;
				} 
				}
				else
				{
				// min mode
				if(outputNextState[i]==OS_MIN)
				{
					// force output high and force state change!
					q= 1 ^ nb;
					oq=nb;
				} else
				if((q!=nb)&&(outputNextState[i]>OS_MIDMIN))
				{
					// force output low and force state change!
					q=nb;
					oq=nb ^ 1;
				}
				}
			
				// now force physical change if a change is needed!
				if(q!=oq)
				{
				//outputState[i]=outputNextState[i];
				if(i==0)OUTPUT1=q; else OUTPUT2=q;
				}

		    } else
		    if((mode & 3)==0)
		    {
			    if(i==0)OUTPUT1=0; else OUTPUT2=0;
			}     
}

/*
byte processFreqVariableUpdate(byte c)
{
	//
	unsigned int j,k;	
	// a small offset from zero is added to account for the interrupt latency.
	switch(fmode[c])
	{
		case 0:
			j=CCPFALL;
			fmode[c]++;
			break;
		case 1:
			j=CCPRISE;
			fmode[c]++;
			break;
		case 2:
			 pperl[c]=j; 
			 pperh[c]=k;
			 fmode[c]++;
			 j=CCPFALL;				
			 break;
		case 3:
 			 fperl[c]=j; 
			 fperh[c]=k;
			 fmode[c]++;
			 j=CCPRISE;
			 break;
		default:
		case 4:
			 j=CCPOFF;
			 break;
	}
	return j;
}
*/

#pragma interrupt myISRHigh
void myISRHigh(void)
{
	byte i;
	unsigned int j0, k0;
	//
   	if(PIR2bits.TMR3IF)
	{
	tmr3w++;
	PIR2bits.TMR3IF=0;
	}

	if(PIR1bits.TMR1IF)
	{
	tmr1w++;
	PIR1bits.TMR1IF=0;
	} 

	if(PIR1bits.CCP1IF & PIE1bits.CCP1IE)
	{
		GBIT=1;
		k0=tmr1w; 
		j0=(unsigned int)((CCPR1H<<8)+CCPR1L); 
	    PIE1bits.CCP1IE=0; 
		switch(fmode[0])
		{
			case 0:
				i=CCPFALL;
				fmode[0]++;
				break;
			case 1:
				i=CCPRISE;
				fmode[0]++;
				break;
			case 2:
				pperl[0]=j0;
				pperh[0]=k0;
				i=CCPFALL;
				fmode[0]++;
				break;
			case 3:
				fperl[0]=j0;
				fperh[0]=k0;
				i=CCPRISE;
				fmode[0]++;
				break;
			case 4:
				i=CCPOFF;
				break;
		}
		tmr1w=0;
		TMR1H=0;
		TMR1L=TMRL_CCP_OFFSET;
		PIR1bits.CCP1IF=0; 	
		CCP1CON=i;
		if(i!=0)PIE1bits.CCP1IE=1; 
    }

	if(PIR2bits.CCP2IF & PIE2bits.CCP2IE)
	{
		GBIT=1;
		k0=tmr3w; 
		j0=(unsigned int)((CCPR2H<<8)+CCPR2L); 
	    PIE2bits.CCP2IE=0; 
		switch(fmode[1])
		{
			case 0:
				i=CCPFALL;
				fmode[1]++;
				break;
			case 1:
				i=CCPRISE;
				fmode[1]++;
				break;
			case 2:
				pperl[1]=j0;
				pperh[1]=k0;
				i=CCPFALL;
				fmode[1]++;
				break;
			case 3:
				fperl[1]=j0;
				fperh[1]=k0;
				i=CCPRISE;
				fmode[1]++;
				break;
			case 4:
				i=CCPOFF;
				break;
		}
		tmr3w=0;
		TMR3H=0;
		TMR3L=TMRL_CCP_OFFSET;
		CCP2CON=i;
		PIR2bits.CCP2IF=0; 	
		if(i!=0)PIE2bits.CCP2IE=1; 
    }
}

#pragma interruptlow myISR
void myISR(void)
{
	int i, j;

	if(UIR & UIE)
	{
	USBDriverService();
	BootService();
	PIR2bits.USBIF=0;
	} 
	
	if(INTCONbits.TMR0IF)
	{
		tmr0Ticks++;
		currentUpdateVariable++;
		if(currentUpdateVariable>=NUM_UPDATE_VARS)
		{ 	
  		 // now beep will be non-zero if at least some beeping is needed...
		 if(reqBeep==0)reqBeep=beep;
		 checkCounter++;
		 if(checkCounter>checkPeriod)
		 { 
			checkCounter=0;
			checkRelayOutput(0);
			checkRelayOutput(1);
		 }	
		 // so beep starts at 0 before all variables are updated, and so beep ends up being correct for all variables after all done...
		 beep=0;
		 currentUpdateVariable=0;   
		 dt=(double)(256-tmr0HPeriod)*(1 << ((T0CON & 0x07)+1));	
    	 dt=dt*FOSC4TP;             // dt is now the period in seconds
    	                         	// a divide by two factor is included in the constant FOSC4TP for the trapezoidal integral calculations!
		}

	    updateVariable(currentUpdateVariable); 
	
		if(fmode[0]>4)
		{
		fmode[0]=0;
		PIE1bits.CCP1IE=0;
		CCP1CON=CCPFALL;
		PIR1bits.CCP1IF=0;
		PIE1bits.CCP1IE=1;
		}

		if(fmode[1]>4)
		{
		fmode[1]=0;
		PIE2bits.CCP2IE=0;
		CCP2CON=CCPFALL;
		PIR2bits.CCP2IF=0;
		PIE2bits.CCP2IE=1;
		}
		TMR0H=tmr0HPeriod;
		TMR0L=1;
		INTCONbits.TMR0IF=0;

	
	}

if(PIR1bits.TMR2IF & PIE1bits.TMR2IE)
	{ 	
		buttonCounter++;
		// note that this interrupt occurs at around 9kHz frequency
		// we are dividing by 256 to get a frequency of button press checking of around 35Hz or so (ie 28ms period) 
		
		if((buttonReq==0)&&((buttonCounter & BUTTON_CHECK_DIVISOR)==0))
		{
			// check the button every BUTTON CHECK DIVISOR (must be a power of 2) loops and if no key press is currently pending...
		    if(RCONbits.POR==0)buttonIndex=0;           // force button system disabled before Power On Reset ack'd	
			switch(buttonIndex)
			{
			// this is the button state machine
			case 0:
				// in the first state, we go to state 1 if the button is not pressed, ie S1=0
				if(S1==0)buttonIndex++;
				break;
			case 1:
				// in this state, we wait till the button is pressed, ie S1=1
				if(S1==1)
				{
				buttonPress[0]=(buttonCounter>>8);
				buttonIndex++;
				} else if(buttonCounter==0)buttonIndex=0;		// this resets the state machine if the counter wraps around.
				break;
			case 2:
				// in this state, we either wait till the button is not pressed, ie S1=0 or 1 second has elapsed...
				if((S1==0)||(((buttonCounter>>8)-buttonPress[0])>SHORT_BUTTON_PRESS_THRESHOLD))
				{
				buttonPress[1]=(buttonCounter>>8);
				buttonIndex++;
				}
				break;
			case 3:
				// we accept the button press in this state
				buttonReq=1;
				break;
			default:
				buttonACK();
				break;
			}
		}
	refreshScreen();
	TMR2=0;
	_asm
		nop
	_endasm
	PIR1bits.TMR2IF=0;
	
	if((OSCCONbits.IDLEN==1)&&(battWakeUpMode!=0))
	{
        _asm
            reset
        _endasm    	   	
    } 
    
	} 
	
	if(INTCON3bits.INT1IF & INTCON3bits.INT1IE)
	{
    	// S1 interrupt used only for waking up from sleep
    	INTCON3bits.INT1IE=0;
        INTCON3bits.INT1IF=0;
        _asm 
            reset
        _endasm
    }   	
}

int printITOA(unsigned long xx, int blanking, int decimal, int mode)
{
		// unsigned integers from 0 to 65535
		// blanking=0 implies zero blanking
		// decimal=   number of digits before the decimal to be printed if 0 then do not print a decimal point (blanking stops after the decimal)
		int i, j;
		unsigned long ux;
		char c;
		
		ux=(unsigned long)xx;
		j=NUM_DIGITS;
		for(i=NUM_DIGITS-1; i>=0; i--)
		{
		c=(char)(0x30+(0x0F&(ux%10)));
		ux=ux/10;
		if((j+1)==decimal){ buffer[j--]='.'; }
		buffer[j--]=c;
		}
		buffer[NUM_DIGITS+1]=0;
		j=j+1;
		if(blanking==0)while((buffer[j]==0x30)&&(j<NUM_DIGITS))j++;
		if(mode==0)if((j>0)&&(buffer[j]=='.'))j--;
		return j;
}

void disWDotsdecimal(unsigned int xx, int font, int blanking)
{
		int i;
		i=printITOA(xx, blanking, 0, font);
		printstringDotsRam(&buffer[i], font, 0);
}

rom double MultiplierF[6]={ 1.0, 10.0, 100.0, 1000.0, 10000.0, 100000.0 };

int printFDecimal(double f, int numdec, int mode)
{
		if(numdec<0)numdec=0; else if(numdec>NUM_DIGITS)numdec=NUM_DIGITS;
		f*=MultiplierF[numdec];
		if(roundingMode!=0)f+=0.5;
		return printITOA(f, 0, (NUM_DIGITS+1)-numdec, mode);
}

void disFDotsdecimal(double f, int numdec, int font, int maxcols)
{
	int i;
	if(f<0.0){ printcharDots(0x2D,font, maxcols); f=-f; }
	if((font==1)&&(f>=10000.0))
	{
		printcharDots(0x3A, 1, 0);
		printcharDots(0x3A, 1, 0);
		printcharDots(0x3A, 1, 0);
		printcharDots(0x3A, 1, 0);
	} else
	{
	i=printFDecimal(f, numdec, 0);
	printstringDotsRam(&buffer[i], font, maxcols);
	}
}

/*
void isetCurrentVariable(int i)
{
	if(i<0)i=0; else if(i>=NUM_VARS)i=NUM_VARS-1;
	currentVariable=i;
}
*/

void clearPolyMemory(void)
{
	int i;
	for(i=0; i<MAX_POLY; i++)
	{
	if(i<(MAX_POLY_INDEXES))
	{ 
	 polyTable[i]=i; 
	 if(i<8)
     {
     	if((i & 1)==0)polyMemory[i]=DEFAULT_ANALOG_SCALE; else polyMemory[i]=1.0;
     } else 
     {
        if((i & 1)==0)polyMemory[i]=DEFAULT_FREQUENCY_SCALE; else polyMemory[i]=DEFAULT_DUTY_SCALE;
     }    
	}
	else 
	polyMemory[i]=0.0;
	}	
}

int getPolyTableAddress(int varn, int index)
{
	return (NUM_RAW*varn+index);
}

int getInterpolationDegree(int varn, int index)
{
	int a;
	a=getPolyTableAddress(varn, index);
	return (polyTable[a+1]-polyTable[a]-1);
}

double getCoefficient(int varn, int index, int degree)
{
	int a;
	a=polyTable[getPolyTableAddress(varn, index)];
	return polyMemory[a+degree];
}

double putCoefficient(int varn, int index, int degree, double value)
{
	int a;
	a=polyTable[getPolyTableAddress(varn, index)];
	polyMemory[a+degree]=value;
	return 0;
}

double computeValue(int var_num, int index, double rw)
{
	double f, g;
	int j;
	int dg;
	int a;
	a=polyTable[NUM_RAW*var_num+index];
	dg=variable[var_num].interpolation[index];
	if(dg==0)
	{
		// special scaling case (implied origin and linear)
		// a scalar transformation only
		g=rw*polyMemory[a];
	}
	else
	{
		g=0.0;					// running total
		f=1.0;
		for(j=dg; j>=0; j--)
		{
		g+=(f*polyMemory[a++]);
		f*=rw;
		}
	}
	return g;
}

int updateVariable(int var_num)
{
	// updates the variable in var_num
	// it returns 0 if not successful (another update in progress) or 1 otherwise
	double f, g, b;
	int i, j, k, m;
	int degree;
		
#if (USE_CLIPPING==1)
	clip(&var_num, 0, (NUM_UPDATE_VARS-1));
#endif
   
    //updatingVariable++;
	if(var_num<NUM_VARS)
	{
     variable[var_num].updated&=~1;
	 switch(variable[var_num].type & 3)
	 {
    	case DEPENDENT_VAR:
					 for(j=0; j<NUM_RAW; j++)
					 {
                     i=variable[var_num].index & 7;
                     if(i==VAR_BATT)
                     {
                     f=battVoltage;                        
                     variable[var_num].updated|=1;                     
                     }    
                     else
                     if(i==VAR_LDR)
                     {
                      f=ldrPercent10/10.0;        
                      variable[var_num].updated|=1;                
                     }    
                     else
                     if(i!=var_num)
                     {
                     if(j==0)
					 k=(variable[var_num].type>>4) & 3; else k=(variable[var_num].type>>6) & 3;
                     // k is the value it depends on!
                     f=variable[i].value[k];  
                     variable[var_num].updated|=(1 & variable[i].updated);
                     }    
                     else f=1.0;
                     variable[var_num].raw[j]=f;
					}
                    break;

    	case SILENT_VAR:
    	            break;
    	            
		case ANALOG_VAR:	
					f=readAN(variable[var_num].index & 3)/1023.0;
					variable[var_num].raw[VALUE_INDEX]=f;
					if(f>0.000970)variable[var_num].raw[DUTY_INDEX]=1.0/f; else variable[var_num].raw[DUTY_INDEX]=1.0;		// note 0.977 = approx 1/1023
					variable[var_num].updated|=1;		
					break;
					
		case FREQUENCY_VAR:
					i=variable[var_num].index & 1;
					if(fmode[i]==4)
					{
					f=(double)fperl[i];
					f+=(double)fperh[i]*65536.0;
					f+=DUTY_CORRECTION_OFFSET;
					g=(double)pperl[i];
					g+=(double)pperh[i]*65536.0;
					g-=DUTY_CORRECTION_OFFSET;
					//
					b=(f+g);
					f=(double)FOSC4_ON_8/b;				// this is the frequency from 0.0 to kHz
					//b=PERCENT_SCALE*(g-TMRL_CCP_OFFSET)/(b-TMRL_CCP_OFFSET_TIMES_2);
					//b=(g-TMRL_CCP_OFFSET)/(b-TMRL_CCP_OFFSET_TIMES_2);
					//b=(g+DUTY_CORRECTION_OFFSET)/b;
					//b=g/b;
					if(f<0.0)f=0.0; 
					else 
					if(f>(FREQ_MAX-1))f=(FREQ_MAX-1);
					//
					variable[var_num].raw[VALUE_INDEX]=f/(double)FREQ_MAX;
					variable[var_num].raw[DUTY_INDEX]=g/b;
					variable[var_num].updated|=1;
				`	fmode[i]++;
        			} 
					break;
	 }
	 // g and f are now the transformed values, so we must scale it now...
	 if(variable[var_num].updated & 1)
	 {	    
	 g=computeValue(var_num, VALUE_INDEX, variable[var_num].raw[VALUE_INDEX]);
	 f=computeValue(var_num, DUTY_INDEX, variable[var_num].raw[DUTY_INDEX]);
	 /*
	 switch(variable[var_num].type & 3)
	 {
		default:
	    case DEPENDENT_VAR:
		case ANALOG_VAR:
	    case SILENT_VAR:
					break;	

		case FREQUENCY_VAR:
					g*=FREQ_MAX;
					break;
	 }
	 */
	 // done...
     i=(variable[var_num].index & 0x80);	// what index (VALUE or DUTY the delta and integral (delta_dt) values apply to)
	 j=(var_num<<1);
  	 if(i==0)
	 {
		variable[var_num].value[DELTA_INDEX]=(g-previousValues[j]);
		variable[var_num].value[DELTA_DT_INDEX]=((g+previousValues[j])*dt);
	 } else
	 {
	    variable[var_num].value[DELTA_INDEX]=(f-previousValues[j+1]);
	    variable[var_num].value[DELTA_DT_INDEX]=((f+previousValues[j+1])*dt);
	 }
	 //
	 variable[var_num].value[DELTA_INDEX]*=variable[var_num].scaling[0];
	 variable[var_num].value[DELTA_DT_INDEX]*=variable[var_num].scaling[1];
	 //
	 if((variable[var_num].mode & MODE_AVG)!=0)
	 { 
	 variable[var_num].value[VALUE_INDEX]+=g; variable[var_num].value[DUTY_INDEX]+=f;
	 variable[var_num].value[VALUE_INDEX]/=2.0; variable[var_num].value[DUTY_INDEX]/=2.0;
	 } else
	 {
	 variable[var_num].value[VALUE_INDEX]=g;
	 variable[var_num].value[DUTY_INDEX]=f;
	 }
	 // work out the integral slice using trapezoidal
	 // rule! Note that different behaviour depending on whether averaging mode is enabled!
	 // note a divide by 2.0 factor is already computed on the dt for optimization purposes ie. speed!
	 variable[var_num].updated|=2;
	 if((variable[var_num].index & 0x40)==0)
	 {
   	 // whether averaging also applies to integral, but only if averaging acquisition is enabled!
	 previousValues[j]=g;               // direct values stored!
     previousValues[j+1]=f;				// direct values stored!
	 } else
	 {
	 previousValues[j]=variable[var_num].value[VALUE_INDEX];				// possibly averaged values stored!
     previousValues[j+1]=variable[var_num].value[DUTY_INDEX];				// possibly averaged values stored!
	 }
	 i=((variable[var_num].type >> 4)& 3);
	 // i is the accumulator channel!
	 variable[var_num].acc+=variable[var_num].value[i];	// accumulate!
	 //
	 checkOutputs(var_num);
  	 }
	 // the following seemingly roundabout way of working updatePeriod out is needed
	 // because tmr0Ticks may change (in the background) while evaluating this!
	 variable[var_num].updatePeriod=variable[var_num].updateTime;
	 variable[var_num].updateTime=tmr0Ticks;
	 variable[var_num].updatePeriod=tmr0Ticks-variable[var_num].updatePeriod;

	 } else
	 {
		// this is to update the other two variables (internal) battery voltage and ldr
		switch(var_num)
		{
			case VAR_LDR:
				ldrLevel+=readAN(ANCHLDR);
				ldrLevel/=2;	       			// divide by 2 (running average)
				f=(ldrLevel-LDRMIN)/LDROFFSET;
				ldrPercent10=f;
				break;
			case VAR_BATT:
				battLevel+=readAN(ANCHBATT);
				battLevel/=2;		     		// divide by 2 (running average)
			/*
				switch(battState)
				{
    				case 0:
    				    if(battLevel>0)battState=1;
    				    break;
    				case 1:
    				    if(battLevel==0)battState=2;
    				    break;
    				default:
    				case 2:
    				    break;
                }		    
			*/
				battVoltage=((double)battLevel*VOLTAGE_RANGE_SCALE);
				checkOutputs(VAR_BATT);
				break;
			default:
				break;
		}
	}
	//updatingVariable--;
	return 1;
}

void initDisplay(void)
{
	int i;
    //
    invertMask=0xFF;
	displayPhase=0;
	displayMode=0;
	for(i=0; i<NUM_DISPLAY_OBJECTS;i++)displayObj[i]=0;
	displayObj[0]=0x80 | VAR_BATT;
	displayObj[1]=0x80 | VAR_LDR;
	buttonACK();
	rimode=0;
	imode=0;
	PORTA=0xFF;
	PORTB=0xFF;
	PORTC=0xFF;
	PORTD=0xFF;
	PORTE=0xFF;
	resetOutputs();
    for(i=0; i<NUM_OUTPUTS; i++){ output[i]=0;  }
	checkCounter=0;
	checkPeriod=DEFAULT_CHECK_PERIOD;
	blankDisplay();
	TRISA=0b00101111;				// A0-A3, A5 inputs rest outputs
	TRISB=0b00000010;				// B1 input the rest outputs
	TRISC=0b00111110;				// C6-C7, C0 outputs the rest inputs
	TRISD=0b00001000;				// D3 an input the rest outputs
	TRISE=0b01111001;				// E1-E2 outputs the rest inputs
	PORTEbits.RDPU=1;				// turn on weak pull ups on PORTD
	minBrightness=DEFAULT_MINBRIGHTNESS;
	maxBrightness=DEFAULT_MAXBRIGHTNESS;
	scrollSpeed=DEFAULT_SCROLL_SPEED;
	roundingMode=DEFAULT_ROUNDINGMODE;
	displayObject=DEFAULT_DISPLAY_OBJECT;
	screenCounter=0;
	PR2=0;
	TMR2=0;
	T2CON=0x4C;				// timer 2 is configured as 1:12 postscale from FOSC4 and expires after 256 giving a frequency at interrupt level of around 7-8kHz
							// note that there are 32 or so phases for the screen refresh giving the screen refresh frequency of 9kHz/32=280Hz
	                        // T2CON =
	                        // bits   7     6 5 4 3         2            1 0
	                        //        X     d3 d2 d1 d0     TMR2ON       prescale=00 = 1:1        d3,d0=D then postscale is divide by (D+1)
	PIR1bits.TMR2IF=0;
	IPR1bits.TMR2IP=0;
	PIE1bits.TMR2IE=1;
}

double readAN(byte channel)
{
	double f;
	unsigned int i, j;
	// returns a number from 0.0 to 1023.0 (or a bit higher depending on ref voltage)
	ADCON0&=0xC3;
	i=0x3C & ((channel & 0x0F)<<2);
	ADCON0|=i;				// set the channel
	i=0;
	f=((double)(battScaling+SCALE_OFFSET)/FIXED_POINT_SCALE);        // scaling factor, this adds to the ADC delay!
	
	//while(i<ADC_DELAY){ i++; }
	ADCON0|=0x02;
	GBIT=1;					// with LED display on, too much current can affect the 5V rail, giving a false reading.
	while((ADCON0 & 0x02)!=0)GBIT=1;	
	i=(unsigned int)((ADRESH<<8) + ADRESL);
	switch(channel)
	{
		case ANCHBATT:
			j=VOLTAGE_CORRECTION_56;
			break;
		case ANCHLDR:
			j=0;
			break;
		default:
			j=VOLTAGE_CORRECTION_22;
			break;
	}
	if(i>=j)i+=j;
	f*=(double)i;
	return f;
}

int initADC(void)
{
	ADCON1=0x09;			// enable AN0 - AN5 as analog inputs the rest as digital pins
	ADCON2=0xBE;			// right justified, FOSC/64 (should not be lower than this, ie faster) and 20 TAD
	ADCON0=0x01;
}

void initCapture(void)
{
	// initialize the CCP1 and CCP2 pins for input capture (to measure frequencies)
	// we use timer 3 for this
	int i;
	T1CON=0;
	T3CON=0;
	tmr3w=0;
	TMR3H=0;
	TMR3L=0;
	tmr1w=0;
	TMR1H=0;
	TMR1L=0;
	T3CON=0xB9;				// 1:8 prescale, CCP1 source is timer 1, CCP2 source is timer 3, both at Fosc/4=12MHz.
	T1CON=0xB1;
	PIR2bits.TMR3IF=0;
	IPR2bits.TMR3IP=1;
	PIE2bits.TMR3IE=1;
	PIR1bits.TMR1IF=0;
	IPR1bits.TMR1IP=1;
	PIE1bits.TMR1IE=1;
	//
	TRISC|=0x06;
	PIR1bits.CCP1IF=0;
	IPR1bits.CCP1IP=1;
	PIR2bits.CCP2IF=0;
	IPR2bits.CCP2IP=1;
	for(i=0; i<FCHANNELS;i++)
	{
	fmode[i]=0;
	fperl[i]=1;
	pperl[i]=1;
	fperh[i]=0;
	pperh[i]=0;
	}
	CCP1CON=CCPFALL;
	CCP2CON=CCPFALL;
	//
	PIE2bits.CCP2IE=1;
	PIE1bits.CCP1IE=1;
}

void initUpdateTimer(void)
{
	tmr0Ticks=0;
	T0CON=DEFAULT_TMR0CON;				// choose 1:8 prescaler, TMR0 16 bit mode, and on.
	INTCONbits.TMR0IF=0;
	INTCON2bits.TMR0IP=0;
	INTCONbits.TMR0IE=1;
}

void initVariables(void)
{
	int i, j;

	invert=0;
	beep=0;
	reqBeep=0;
	beepCounter=0;
	limitCounter=0;
	limitObj=0;
	limitPeriod=DEFAULT_LIMIT_PERIOD;
	beepPeriod=DEFAULT_BEEP_PERIOD;
	currentUpdateVariable=0;
	clearPolyMemory();
	zeroAcc=1;
	for(i=0; i<NUM_VARS;i++)
	{
	for(j=0; j<NUM_VALUES;j++)
	{ 
	variable[i].value[j]=0.0; 
	if(i<4){
    	variable[i].max[j]=DEFAULT_ANALOG_SCALE;
	    variable[i].min[j]=0.0;
	} else
	{
    	variable[i].max[j]=FREQ_MAX;
    	variable[i].min[j]=0.0;
    }   	    
	if(j<NUM_RAW)
	{ 
	variable[i].raw[j]=0.0; 
	} 
	}
	for(j=0; j<NUM_RAW; j++)variable[i].decp[j]=DEFAULT_SYSTEMDECP;
	for(j=0; j<NUM_SCALING; j++)variable[i].scaling[j]=1.0;
	if(i<4)
		{ 
		variable[i].type=ANALOG_VAR; 
		variable[i].index= i & 3;
		}
		else 
		{
		variable[i].type=FREQUENCY_VAR;
		variable[i].index= i & 1;
		}
	variable[i].mode=DEFAULT_MODE;
	for(j=0; j<NUM_STRINGS_PER_VAR;j++)variable[i].name[j]=0;
	variable[i].updateTime=0;
	variable[i].updatePeriod=0;
	variable[i].updated=0;
	variable[i].invert=0;
	variable[i].hysteresis=DEFAULT_HYSTERESIS;
	for(j=0; j<NUM_RAW;j++)variable[i].interpolation[j]=getInterpolationDegree(i,j);
	}
	initUpdateTimer();
}

void initSystem(void)
{
	minorDelay=MINOR_DELAY_MS;
	majorDelay=MAJOR_DELAY_MS;
	verhigh=VER_HIGH;
	verlow=VER_LOW;
	pushing=0;
	pushVariable=0;
	szVariable=sizeof(VARIABLE);
	addressVariableFlash=(int)&VariableData[0];
	addressStringFlash=(int)&Strings[0];
	maxStringSize=AVG_STRING_SIZE;
	stringsSize=SIZE_OF_STRINGS;				// total String Memory in Flash (in bytes)
	INTCON2bits.RBPU=0;
	initVariables();
	initADC();
	initDisplay();
	initCapture();
	INTCON3bits.INT1IP=0;   // low priority for INT1 external interrupt (S1) only used to wake up from sleep!
	INTCON3bits.INT1IF=0;
	INTCON2bits.INTEDG1=1;  // trigger on rising edge (button S1 line normally low)
	INTCON3bits.INT1IE=0;   // disabled for now
	RCONbits.IPEN=1;		// enable prioritized interrupts!
	INTCONbits.GIEL=1;		// enable low interrupts!
}

/*
float readVCC()
{
	// assumes that the output of the 3.3V internal regulator for USB
	// is connected via a 33k ohm resistor to the AN0 input
	float t;
	int i;
	i=readAN(0);
	if(i>0)
	{ 
		t=usbRegAssumedVoltage*1.023000/((float)i);
	} 
	else
	{
		t=0.0;
	}		
	return t;
}

*/

void USBConnect(void)
{
	mInitializeUSBDriver();     // See usbdrv.h
 	USBCheckBusStatus();        // Modified to always enable USB module
}

/*
void updateAllSettings(void)
{

}

void loadDefaults(void)
{

}
*/

void zeroAccumulators(void)
{
    // zero accumulators
    int i;
    //
	for(i=0; i<NUM_VARS; i++)
	{
			if(variable[i].acc!=0.0)
			{
			variable[i].acc=0.0;
			putVariable(i);
			}
	}
}	

int initSequence(void)
{
	int i;
	// restore settings from non volatile memory if previously saved...
	//updatingVariable=0;
	inHighInterrupt=0;
	systemName=SIZE_OF_STRINGS;	                // default value!
	battScaling=DEFAULT_BATTSCALING;
	battMin=DEFAULT_BATTMIN;
	battMax=DEFAULT_BATTMAX;
	battVoltage=0.0;
	systemDecp=DEFAULT_SYSTEMDECP;
	tmr0Con=DEFAULT_TMR0CON;
	tmr0ConLog=DEFAULT_TMR0CONLOG;
	tmr0HPeriod=DEFAULT_TMR0HPERIOD;
	fosc4div1000=(int)(FOSC4/1000.0);
	ldrLevel=0;
	battLevel=0;
	initSystem();
	IPR2bits.USBIP=USBPRIORITY;	   // select interrupt priority (added MG)
	UIR=0;
	PIR2bits.USBIF=0;
	PIE2bits.USBIE=1;						// enable USB interrupts!	(added MG)
	i=ReadEEPROM(MAGIC_VALUE_ADR);
	if(i!=MAGIC_VALUE)
	{
		// there are no previously stored settings so save defaults!
		DelayMs(POWERON_DELAY);				// this is a delay for stabilising the power supply before writing to Flash.
	    if(zeroAcc!=0)zeroAccumulators();
	    saveAllSettings();
		WriteEEPROM(MAGIC_VALUE_ADR, MAGIC_VALUE);
	} else
	{
		// there are previously stored settings so retrieve them...
		DelayMs(POWERON_DELAY);
		restoreAllSettings();
		if(zeroAcc!=0)zeroAccumulators();
	}			
	INTCONbits.GIE=INTERRUPT_ENABLE;		// enable all interrupts!!
}

#pragma romdata

byte* persistentDataRamPointers[NUM_PERSISTENT_SETTINGS]={
&minBrightness,
&maxBrightness,
&scrollSpeed,
&displayObject,
&displayMode,
&tmr0Con,
&tmr0ConLog,
&tmr0HPeriod,
&roundingMode,
&systemDecp,
&battScaling,
&battMin,
&battMax,
&systemName,
&zeroAcc,
&output[0],
&checkPeriod,
&beepPeriod,
&minorDelay,
&majorDelay,
&invertMask,
&battSleepMode,
&battWakeUpMode,
&limitPeriod,
&displayObj[0]
};

byte persistentDataSizes[NUM_PERSISTENT_SETTINGS]={ 
sizeof(minBrightness),
sizeof(maxBrightness),
sizeof(scrollSpeed),
sizeof(displayObject),
sizeof(displayMode),
sizeof(tmr0Con),
sizeof(tmr0ConLog),
sizeof(tmr0HPeriod),
sizeof(roundingMode),
sizeof(systemDecp),
sizeof(battScaling),
sizeof(battMin),
sizeof(battMax),
sizeof(systemName),
sizeof(zeroAcc),
NUM_OUTPUTS,
sizeof(checkPeriod),
sizeof(beepPeriod),
sizeof(minorDelay),
sizeof(majorDelay),
sizeof(invertMask),
sizeof(battSleepMode),
sizeof(battWakeUpMode),
sizeof(limitPeriod),
NUM_DISPLAY_OBJECTS
};

void processPersistentSettings(byte dir)
{
	// read dir or write dir
	byte* ptr;
	byte size;
	int i, j;

	i=START_PERSISTENT_ADR;
	j=0;
	while(j<NUM_PERSISTENT_SETTINGS)
	{
	ptr=persistentDataRamPointers[j];
	size=persistentDataSizes[j];
	doRWEE(i, ptr, size, dir);
	i+=size;
	j++;
	}
}

void saveAllSettings(void)
{
	int i;
	for(i=0; i<NUM_VARS;i++)
	{
	putVariable(i);
	}
	putPolyData();
	processPersistentSettings(WRITE_DIR);


//	bsetVar(minBrightness, &minBrightness, ABSMIN_BRIGHTNESS, ABSMAX_BRIGHTNESS, MINBRIGHTNESS_ADR);
//	bsetVar(maxBrightness, &maxBrightness, ABSMIN_BRIGHTNESS, ABSMAX_BRIGHTNESS, MAXBRIGHTNESS_ADR);
//	bsetVar(scrollSpeed, &scrollSpeed, MINSCROLL_SPEED, MAXSCROLL_SPEED, SCROLLSPEED_ADR);
//	bsetVar(displayObject, &displayObject, 0, NUM_DISPLAY_OBJECTS-1, DISPLAYOBJECT_ADR);
//	bsetVar(displayMode, &displayMode, MINDISPLAY_MODE, MAXDISPLAY_MODE, DISPLAYMODE_ADR);
//	bsetVar(tmr0Con, &tmr0Con, LOT0CON, HIT0CON, TMR0CON_ADR);
//	bsetVar(tmr0ConLog, &tmr0ConLog, LOT0CON, HIT0CON, TMR0CON_ADR);
//	bsetVar(tmr0HPeriod, &tmr0HPeriod, MINTMR0HPERIOD, MAXTMR0HPERIOD, TMR0HPERIOD_ADR);
//	bsetVar(roundingMode, &roundingMode, MINROUNDINGMODE, MAXROUNDINGMODE, ROUNDINGMODE_ADR);
//	bsetVar(systemDecp, &systemDecp, MINSYSTEMDECP, MAXSYSTEMDECP, SYSTEMDECP_ADR);

/*
	WriteEEPROM(MINBRIGHTNESS_ADR, minBrightness);
	WriteEEPROM(MAXBRIGHTNESS_ADR, maxBrightness);
	WriteEEPROM(SCROLLSPEED_ADR, scrollSpeed);
	WriteEEPROM(DISPLAYOBJECT_ADR, displayObject);
	WriteEEPROM(DISPLAYMODE_ADR, displayMode);
	WriteEEPROM(TMR0CON_ADR, tmr0Con);
	WriteEEPROM(TMR0CONLOG_ADR, tmr0ConLog);
	WriteEEPROM(TMR0HPERIOD_ADR, tmr0HPeriod);
	WriteEEPROM(ROUNDINGMODE_ADR, roundingMode);
	WriteEEPROM(SYSTEMDECP_ADR, systemDecp);
	WriteEEPROM(BATTSCALINGL_ADR, battScaling);
	WriteEEPROM(BATTMAX_ADR, battMax);
	WriteEEPROM(BATTMIN_ADR, battMin);
	WriteEEPROM(SYSNAMEH_ADR, (systemName>>8));
	WriteEEPROM(SYSNAMEL_ADR, (systemName));
	WriteEEPROM(OUTPUT1_ADR, output[0].mode);
	WriteEEPROM(OUTPUT2_ADR, output[1].mode);
	WriteEEPROM(ZEROACC_ADR, zeroAcc);
	for(i=0; i<NUM_DISPLAY_OBJECTS;i++)
	{
	WriteEEPROM(DISPLAY_OBJS_ADR+i, displayObj[i].mode);
	}
*/

}

void restoreAllSettings(void)
{
	int i;
	for(i=0; i<NUM_VARS;i++)
	{
	getVariable(i);
	}
	getPolyData();
	processPersistentSettings(READ_DIR);

/*
	minBrightness=ReadEEPROM(MINBRIGHTNESS_ADR);
	maxBrightness=ReadEEPROM(MAXBRIGHTNESS_ADR);
	scrollSpeed=ReadEEPROM(SCROLLSPEED_ADR);
	displayObject=ReadEEPROM(DISPLAYOBJECT_ADR);
	displayMode=ReadEEPROM(DISPLAYMODE_ADR);
	tmr0Con=ReadEEPROM(TMR0CON_ADR);
	tmr0ConLog=ReadEEPROM(TMR0CONLOG_ADR);
	roundingMode=ReadEEPROM(ROUNDINGMODE_ADR);
	systemDecp=ReadEEPROM(SYSTEMDECP_ADR);
	zeroAcc=ReadEEPROM(ZEROACC_ADR);
	output[0].mode=ReadEEPROM(OUTPUT1_ADR);
	output[1].mode=ReadEEPROM(OUTPUT2_ADR);
//	bsetVar(ReadEEPROM(MINBRIGHTNESS_ADR), &minBrightness, ABSMIN_BRIGHTNESS, ABSMAX_BRIGHTNESS, MINBRIGHTNESS_ADR);
//	bsetVar(ReadEEPROM(MAXBRIGHTNESS_ADR), &maxBrightness, ABSMIN_BRIGHTNESS, ABSMAX_BRIGHTNESS, MAXBRIGHTNESS_ADR);
//	bsetVar(ReadEEPROM(SCROLLSPEED_ADR), &scrollSpeed, MINSCROLL_SPEED, MAXSCROLL_SPEED, SCROLLSPEED_ADR);
//	bsetVar(ReadEEPROM(DISPLAYOBJECT_ADR), &displayObject, 0, NUM_DISPLAY_OBJECTS-1, DISPLAYOBJECT_ADR);

//	bsetVar(ReadEEPROM(DISPLAYMODE_ADR), &displayMode, MINDISPLAY_MODE, MAXDISPLAY_MODE, DISPLAYMODE_ADR);
//	bsetVar(ReadEEPROM(TMR0CON_ADR), &tmr0Con, LOT0CON, HIT0CON, TMR0CON_ADR);
//	bsetVar(ReadEEPROM(TMR0HPERIOD_ADR), &tmr0HPeriod, MINTMR0HPERIOD, MAXTMR0HPERIOD, TMR0HPERIOD_ADR);
//	bsetVar(ReadEEPROM(TMR0CONLOG_ADR),&tmr0ConLog, LOT0CON, HIT0CON, TMR0CONLOG_ADR);
//	bsetVar(ReadEEPROM(ROUNDINGMODE_ADR), &roundingMode, MINROUNDINGMODE, MAXROUNDINGMODE, ROUNDINGMODE_ADR);
//	bsetVar(ReadEEPROM(SYSTEMDECP_ADR), &systemDecp, MINSYSTEMDECP, MAXSYSTEMDECP, SYSTEMDECP_ADR);
	battScaling=ReadEEPROM(BATTSCALINGH_ADR)*256.0 + ReadEEPROM(BATTSCALINGL_ADR);
	battScaling=ReadEEPROM(BATTSCALINGL_ADR);
	battMin=ReadEEPROM(BATTMIN_ADR);
	battMax=ReadEEPROM(BATTMAX_ADR);
	systemName=ReadEEPROM(SYSNAMEH_ADR)<<8;
	systemName+=ReadEEPROM(SYSNAMEL_ADR);
	for(i=0; i<NUM_DISPLAY_OBJECTS;i++)
	{
	displayObj[i].mode=ReadEEPROM(DISPLAY_OBJS_ADR+i);
	}
*/
	
}

/*
void blink(int times, int iinvert)
{
	int i;
	i=0;
	while(i<times)
	{
	invert=iinvert;
	DelayMs(MINOR_DELAY_MS);
	invert=0;
	DelayMs(MAJOR_DELAY_MS);
	i++;
	}
}
*/

void displayDisplayObject(int disnum, int dismode)
{
	int i, ivar;
	double f;
	byte decp;
	byte* updated;
	byte x;
	byte targetinvert;
	//
	if((disnum>=0)&&(disnum<(NUM_DISPLAY_OBJECTS)))
	{
	 if((displayObj[disnum] & 0x80)||(disnum==0))			// object 0 is always enabled!
	 {
	 ivar=displayObj[disnum] & 0x0F;
	 if(ivar>=NUM_VARS)
	 {
	 x=0xFF;
	 targetinvert=0;
	 switch(ivar)
	 {
		case VAR_BATT:
			f=battVoltage;
			targetinvert=battInv;
			break;
		case VAR_LDR:
			f=ldrPercent10/10.0;
			break;
		default:
			f=0.0;
			break;
  	 }
	 decp=systemDecp;
	 updated=&x;
	 i=0;
	 }
	 else
	 {
	 if((displayObj[disnum] & 0x40)==0)
	 {
	 // normal display!
	 i=(displayObj[disnum]>>4)&3;
	 f=variable[ivar].value[i];
	 targetinvert=((variable[ivar].invert)>>(i<<1)) & 3;	
	 } else
	 {
	 // accumulator display!
	 i=((variable[ivar].index>>4)& 3);
	 // i is the channel!
	 f=variable[ivar].acc;
	 targetinvert=0;
	 }
	 if(i<NUM_RAW)decp=variable[ivar].decp[i]; else decp=systemDecp;
	 updated=&variable[ivar].updated;
	 }
	 // f is now the value to display
	 invert=0;
	 if(dismode==0)
	{
    	
	printString(ivar, i, NAME_INDEX, 0);
    if((variable[ivar].type & 3)<SILENT_VAR)
    {
	 if((displayObj[disnum] & 0x40)==0)
	 {
	 printstringDots((rom char*)":",0,0,'\0');
	 } else
	 {
	 // accumulated display!
	 printstringDots((rom char*)"(+):",0,0,'\0');
	 }
	 if(((*updated) & 2)!=0)
	 {
	 disFDotsdecimal(f,decp,0,0); 
	 x=(*updated)&~2;
	 *updated=x;
	 }
	 else printstringDots((rom char*)"N/A",0,0,'\0');
	printString(ivar,i,UNIT_INDEX,0);
    }
	printstringDots((rom char*)"   ",0,0,'\0');
	} 
	 else
	 {
	  cursor=-1;
	  if(((*updated) & 2)!=0)
	  {
	  disFDotsdecimal(f,decp,1,15); 
	  x=(*updated)&~2;
	  *updated=x;
	  } 
	  else 
	  {
	  printstringDots((rom char*)"----",1,15,'\0');
	  }
	  clearUpTo(cursor, DISPLAY_COLUMNS);
	}
	
	if(dismode!=0)invert=targetinvert; else invert=0;
	}
	Delay16Ms(minorDelay);
	invert=0;
	Delay16Ms(majorDelay);
  	} 
	else
	{
	// outside of range of displayObject index so the display is off!
	clearUpTo(0,15);
	cursor=(DISPLAY_COLUMNS-1);
	}
//	blink(1, invert);
}

int getNextDisplayObject(int start)
{
	int i;
	i=0;
	while(i==0)
	{
	start++;
	if(start>NUM_DISPLAY_OBJECTS)start=0;
	if((start==0)||(start==NUM_DISPLAY_OBJECTS)||(displayObj[start] & 0x80))i=1;
	}
	return start;
}

void printChangeTitle(void)
{
	if(displayObject<(NUM_DISPLAY_OBJECTS))
	{
	printstringDots((rom char*)" >",0,0,'\0');
	displayDisplayObject(displayObject,0);
	}
	else
	{ 
	resetOutputs();
	printstringDots((rom char*)">Off",0,0,'\0');  
	}
}
				
void checkUSBSense(void)
{
	if(usb_sense != (1 & (USBSENSE)))
			    {
				// a USB cable has been connected or disconnected!
				printstringDots((rom char*)">USB ",0,0,'\0');
				if(USBSENSE==USBSENSEPOLARITY)printstringDots((rom char*)"On!   ",0,0,'\0'); else printstringDots((rom char*)"Off!   ",0,0,'\0');
				usb_sense=(1 & (USBSENSE));
	    		}
}
	
void goToSleep(void)
{
        
		PORTA=0xFF;
	    PORTB=0xFF;
	    PORTC=0xFF;
	    PORTD=0xFF;
	    PORTE=0xF9;            // this turns off the two relay outputs too!
	    blankDisplay();
   	    GBIT=1;
        
		INTCON=0;
        PIR1=0;
        PIR2=0;
        PIE1=0;
        PIE2=0;
        INTCON2bits.INTEDG1=1;  // trigger on rising edge (button S1 line normally low)
        INTCON3bits.INT1IP=0;
        INTCON3bits.INT1IF=0;
        INTCON3bits.INT1IE=1;   // turn on for wake up on S1 pressed, INT1 interrupt!
        INTCONbits.GIEL=1;
        
        /*
        i=0;
    	while((S1==0)&&(i<OVERRIDE_LOWBATT_DELAYMS))
    	{
        	DelayMs(10);
        	i++;
        } 
        */  	
        if(battWakeUpMode!=0)
        {
            TMR2=0;
            PR2=0xFF;
        	T2CON=0x7F;		// timer 2
        	PIR1bits.TMR2IF=0;
	        IPR1bits.TMR2IP=0;
	        PIE1bits.TMR2IE=1;
        } 
        INTCONbits.GIE=1;
		if(USBSENSE!=USBSENSEPOLARITY)
		{
		RCONbits.POR=0;         // force POR on restart! must press S1!    	
		if(battWakeUpMode!=0)OSCCON=0x82;            // go to idle RC run mode!
    	else OSCCON=0;
    	_asm
    	    sleep
    	_endasm 
		}
}

int isBatteryLow(void)
{
    // returns 0 if not low or non zero if lower than minimum set... 
    int i;
    // do not call this function repeatedly, while in main loop!
    battLevel=readAN(ANCHBATT);
	battVoltage=((double)battLevel*VOLTAGE_RANGE_SCALE);
	if(battVoltage<(battMin*BATTSCALE))return 1;
	return 0;
}    

void main(void)
{
	int i,j, key, idis;
	double fx;
	int freezePointChannel;
	char* ptr;

	reqReset=0;
    INTCONbits.GIE=0;
	OSCCON=0;
	battSleepMode=DEFAULTBATTSLEEPMODE;         // default batt state mode
	battLow=battSleepMode;
	battWakeUpMode=DEFAULTBATTWAKEUPMODE;
	initSequence();
	T0CON=tmr0Con;
	idis=0;	
	localReqBeep=0;
	//
	USBConnect();
	adjustBrightnessLDR();
	//
	if(RCONbits.POR==1)
	{
    	// not a power on reset...
	    printstringDots((rom char*)">Reset   ",0,0,'\0');
	} else 
	{ 
		if(battSleepMode!=0)
    	{
			if((isBatteryLow())!=0)
			{
    		// can be overriden with S1 pressed! (and doesn't sleep if USB connected)
    		DelayMs(200);   // for debouncing!
    		i=0;
    		while((S1==1)&&(i<OVERRIDE_LOWBATT_DELAYMS))
    		{
        		i++;
        		DelayMs(1);
            }  		
    		if(USBSENSE!=USBSENSEPOLARITY)
    		{
        		// if there's no USB power either, then 
        		if(i==OVERRIDE_LOWBATT_DELAYMS)
        		{
            		// if here then button was pressed for at least the override delay in ms continuously, so override the go to Sleep function...
            		battLow=battSleepMode;
            		buttonACK();
                } else 
				{
					goToSleep();
				}
            }  		
			} 
        }   	
        //
        RCONbits.POR=1;     // acknowledge power on reset, so buttons become active...
        if(displayObject<(NUM_DISPLAY_OBJECTS))
		{
		       if((systemName>=0)&&(systemName<SIZE_OF_STRINGS))
			   {
			   ptr=&Strings[systemName]; 
			   ptr=getStringSubIndex(ptr, 4);
	           printstringDots(ptr,0,0, STRING_DELIMITER);
     		   } else
			   {	   
			   printstringDots((rom char*)">Car Scrolling Display ver:",0,0,'\0');
			   fx=verhigh + verlow/100.0;
			   disFDotsdecimal(fx, 2, 0, 0);
			   printstringDots((rom char*)"   ",0,0,'\0');
			   }
		}
	}	
	//
	buttonACK();
	if((displayMode<2)&&(displayObject<(NUM_DISPLAY_OBJECTS)))printChangeTitle();
	//
    if(USBSENSE==0)usb_sense=0; else usb_sense=1;
	//
	//
	while(1)
	{
			if(reqReset!=0)
			{
				_asm
					reset
				_endasm
			}
		
		    if((battSleepMode!=0)&&(battLow==0)&&(USBSENSE!=USBSENSEPOLARITY))
		    {
    		 // battSleepMode !=0 requires a different power supply connection.
    		 // in this case, we write settings to flash and go to sleep mode to save power
    		 // (because this mode requires an omnipresent batt power connection.
    		 printstringDots((rom char*)">Low ",0,0,'\0');
    		 printString(VAR_BATT, 0, 0, 0);
    		 printstringDots((rom char*)"! Shutting Down...   ",0,0,'\0');
    		 // save all settings prior to going to sleep, useful for saving accumulators if not in zeroacc mode!
    		 saveAllSettings();
             goToSleep();
    		}    
			adjustBrightnessLDR();
			//
			if((displayMode==1)&&(displayObject<(NUM_DISPLAY_OBJECTS)))
			{
			// beeps work only in mode 1
			switch(beepCounter)
				{
				
				case 0:
					
					if((localReqBeep & 0x03)!=0)OUTPUT1=1; 
					if((localReqBeep & 0x30)!=0)OUTPUT2=1;
					break;
					
				case 1:
					if((localReqBeep & 0x02)!=0)OUTPUT1=0;
					if((localReqBeep & 0x20)!=0)OUTPUT2=0;
					break;
				
				case 2:
					if((localReqBeep & 0x02)!=0)OUTPUT1=1;
					if((localReqBeep & 0x20)!=0)OUTPUT2=1;
					if((localReqBeep & 0x01)!=0)OUTPUT1=0;
					if((localReqBeep & 0x10)!=0)OUTPUT2=0;
					break;
				
				case 3:
					if((localReqBeep & 0x06)!=0)OUTPUT1=0;
					if((localReqBeep & 0x60)!=0)OUTPUT2=0;
					break;
				
	    		default:
	    			// so as not to interfere with the beeps!
					checkUSBSense();
					break;
				}
			
			
			beepCounter++;
			if(beepCounter>(CRITICAL_BEEP+beepPeriod))
			{
				localReqBeep=reqBeep;
				reqBeep=0;
				beepCounter=0;
			}
															
			} else
			{
    		    beepCounter=0;
    			checkUSBSense();
			
			    if(displayObject<(NUM_DISPLAY_OBJECTS))
			    {
    		        limitCounter++;
    		        if(limitCounter>limitPeriod)
    		        {
                        limitObj=getNextDisplayObject(limitObj);
                        if(limitObj<NUM_DISPLAY_OBJECTS)
                        {
                        if((displayObj[limitObj] & 0x40)==0)
        		        {
        		         i=displayObj[limitObj] & 0x0F;
        		         j=(displayObj[limitObj] >> 4) & 3;
        		         // reuse limitCounter to minimize RAM use!
        		         if((i<NUM_VARS)||(i==VAR_BATT))
        		         {
        		         if(i<NUM_VARS)limitCounter=(((variable[i].invert)>>(j<<1)) & 3); else { limitCounter=battInv; j=0; }
        		         if(limitCounter!=0)
        		         {
            		     printstringDots((rom char*)">", 0,0, '\0');
            		     printString(i, j, VALUE_INDEX, 0);
            		     if(limitCounter==1)printstringDots((rom char*)" Too High!    ",0,0,'\0');
            		     else printstringDots((rom char*)" Too Low!    ",0,0,'\0');
            		     }
            		     limitCounter=0;
            		     } else limitCounter=limitPeriod;
            		    } else limitCounter=limitPeriod;
            		    } else limitCounter=limitPeriod;
        		    }         
    			}    
			}
			
			//printstringDots((rom char*)"beepC:",0,0,'\0');
			//disWDotsdecimal(beepCounter,0,0);
			//printstringDots((rom char*)"locRq:",0,0,'\0');
			//disWDotsdecimal(localReqBeep,0,0);
			invert=0;
	    	if(pushing==0)
			{
				key=NO_PRESS;
				if(buttonReq==1)
				{
				// note buttonPress are in units of around 40ms so 1 second is equal to 25
				i=(buttonPress[1]-buttonPress[0]);
				if(i<SHORT_BUTTON_PRESS_THRESHOLD)key=SHORT_PRESS; else key=LONG_PRESS;
				}				
				
			    if(displayMode<2)
				{
					if(key==LONG_PRESS)
					{
						if(displayObject!=NUM_DISPLAY_OBJECTS)
						{
						displayMode++;
						if(displayMode>MAXDISPLAY_MODE)displayMode=0;
						WriteEEPROM(DISPLAYMODE_ADR, displayMode);
						buttonACK();
						if(displayMode==0)printChangeTitle();
						} else
						{
						displayObject=getNextDisplayObject(displayObject);
						WriteEEPROM(DISPLAYOBJECT_ADR, displayObject);
						buttonACK();
						printChangeTitle();
						}
						idis=0;
					} 
					else
					if(key==SHORT_PRESS)
					{
					displayObject=getNextDisplayObject(displayObject);
					WriteEEPROM(DISPLAYOBJECT_ADR, displayObject);
					invert=0;
					buttonACK();
					printChangeTitle();
					} 
				}
				else	
				{
					if(key!=NO_PRESS)
					{
					displayMode=0;
					WriteEEPROM(DISPLAYMODE_ADR, displayMode);
					buttonACK();
					}
				}
			
				if((rimode==0)&&(imode==0))
				{
					if(displayMode<2)
					{
					displayDisplayObject(displayObject, displayMode);
					}
					else
					{
					if(idis==0)
					{
					printstringDots((rom char*)">All   ",0,0,'\0'); 
					}
					displayDisplayObject(idis, 0);
					idis=getNextDisplayObject(idis);
					if(idis==NUM_DISPLAY_OBJECTS)idis=0;
					}
				}
				else
				if((rimode & 0x80)!=0)
				{
					// service imode request change ...
					rimode&=0x7F;
					blankDisplay();
					if(rimode==LOGGING_MODE)
					{ 
					printstringDots((rom char*)">Logging   ",0,0,'\0');	
					imode=rimode;
					PIE1bits.TMR2IE=0; 
					T0CON=tmr0ConLog; 
					} 
					else
					if(rimode==FLASHWRITE_MODE)
					{
					imode=rimode;	
					} else	
					{ 
					imode=0; 
					T0CON=tmr0Con; 
					blankDisplay();
					PIE1bits.TMR2IE=1; 
					}
					rimode=0;
				} else
				if(imode==FLASHWRITE_MODE)
				{
    				while((reqReset==0)&&((rimode & 0x80)==0))printstringDots((rom char*)">Wait    ",0,0,'\0');
    				resetOutputs();
				}	
			} 
			else
			{
            resetOutputs();
			invert=0;
			cursor=0;
#if (USE_CLIPPING==1)
     		clip(&pushVariable,1,NUM_VARS);
#endif
			while(pushVariable==0);
			freezePointChannel=(pushVariable-1);
			pushIndex=(pushIndex & (NUM_RAW-1));
			while(pushing==1);
			pushing=2;
			printstringDots((rom char*)"   >Calibrating: ",0,0,'\0');
			printString(freezePointChannel, pushIndex, NAME_INDEX, 0);
			printstringDots((rom char*)"(",0,0,'\0');
			printString(freezePointChannel, pushIndex, UNIT_INDEX, 0);
			printstringDots((rom char*)")   ",0,0,'\0');
			i=0;
			while((pushing!=0)&&(i<pushCount))
			{
			printstringDots((rom char*)">Pt.",0,0,'\0');
			disWDotsdecimal(pushStart+i+1, 0, 0);
			printstringDots((rom char*)"   ",0,0,'\0');
			pushing=2;
			while(pushVariable!=0)
			{ 		
			invert=0;
			cursor=-1;
			disFDotsdecimal(variable[freezePointChannel].value[pushIndex],4,1,15); 
			Delay16Ms(minorDelay);
			invert=2;					// choose flashing for calibration!
			Delay16Ms(minorDelay);
			}
			invert=0;
			cursor=-1;
			disFDotsdecimal(variable[freezePointChannel].value[pushIndex],4,1,15);
			rimode=0;
			pushing=3;
			while(pushVariable==0);
			if(pushing==3)i++;
			}
			pushing=5;			
			printstringDots((rom char*)"   >Done.   ",0,0,'\0');
			pushVariable=0;
      		beep=0;
      		localReqBeep=0;
      		reqBeep=0;
      		beepCounter=0;
			while(pushing==5);
			pushing=0;
			buttonACK();
			}// end of push section
	}// end while loop
	}
//
//#pragma code user = RM_RESET_VECTOR
/** EOF main.c ***************************************************************/
